<?php
/**
 * Custom icons for this theme.
 *
 * @package WordPress
 * @subpackage Twenty_Twenty
 * @since 1.0.0
 */

if ( ! class_exists( 'TwentyTwenty_SVG_Icons' ) ) {
	/**
	 * SVG ICONS CLASS
	 * Retrieve the SVG code for the specified icon. Based on a solution in Twenty Nineteen.
	 */
	class TwentyTwenty_SVG_Icons {
		/**
		 * GET SVG CODE
		 * Get the SVG code for the specified icon
		 *
		 * @param string $icon Icon name.
		 * @param string $group Icon group.
		 * @param string $color Color.
		 */
		public static function get_svg( $icon, $group = 'ui', $color = '#1A1A1B' ) {
			if ( 'ui' === $group ) {
				$arr = self::$ui_icons;
			} elseif ( 'social' === $group ) {
				$arr = self::$social_icons;
			} else {
				$arr = array();
			}
			if ( array_key_exists( $icon, $arr ) ) {
				$repl = '<svg class="svg-icon" aria-hidden="true" role="img" focusable="false" ';
				$svg  = preg_replace( '/^<svg /', $repl, trim( $arr[ $icon ] ) ); // Add extra attributes to SVG code.
				$svg  = str_replace( '#1A1A1B', $color, $svg ); // Replace the color.
				$svg  = str_replace( '#', '%23', $svg ); // Urlencode hashes.
				$svg  = preg_replace( "/([\n\t]+)/", ' ', $svg ); // Remove newlines & tabs.
				$svg  = preg_replace( '/>\s*</', '><', $svg ); // Remove white space between SVG tags.
				return $svg;
			}
			return null;
		}

		/**
		 * GET SOCIAL LINK SVG
		 * Detects the social network from a URL and returns the SVG code for its icon.
		 *
		 * @param string $uri The URL to retrieve SVG for.
		 */
		public static function get_social_link_svg( $uri ) {
			static $regex_map; // Only compute regex map once, for performance.
			if ( ! isset( $regex_map ) ) {
				$regex_map = array();
				$map       = &self::$social_icons_map; // Use reference instead of copy, to save memory.
				foreach ( array_keys( self::$social_icons ) as $icon ) {
					$domains            = array_key_exists( $icon, $map ) ? $map[ $icon ] : array( sprintf( '%s.com', $icon ) );
					$domains            = array_map( 'trim', $domains ); // Remove leading/trailing spaces, to prevent regex from failing to match.
					$domains            = array_map( 'preg_quote', $domains );
					$regex_map[ $icon ] = sprintf( '/(%s)/i', implode( '|', $domains ) );
				}
			}
			foreach ( $regex_map as $icon => $regex ) {
				if ( preg_match( $regex, $uri ) ) {
					return twentytwenty_get_theme_svg( $icon, 'social' );
				}
			}
			return null;
		}

		/**
		 * ICON STORAGE
		 * Store the code for all SVGs in an array.
		 *
		 * @var array
		 */
		public static $ui_icons = array(
			'arrow-down'          => '<svg xmlns="http://www.w3.org/2000/svg" width="22" height="24" viewBox="0 0 22 24">
		<polygon fill="#FFF" points="721.105 856 721.105 874.315 728.083 867.313 730.204 869.41 719.59 880 709 869.41 711.074 867.313 718.076 874.315 718.076 856" transform="translate(-709 -856)"/>
		</svg>',
			'arrow-down-circled ' => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32">
		<path fill="#FFF" fill-rule="evenodd" d="M16,32 C7.163444,32 0,24.836556 0,16 C0,7.163444 7.163444,0 16,0 C24.836556,0 32,7.163444 32,16 C32,24.836556 24.836556,32 16,32 Z M16.7934656,8 L15.4886113,8 L15.4886113,21.5300971 L10.082786,16.1242718 L9.18181515,17.0407767 L16.1410384,24 L23.1157957,17.0407767 L22.1915239,16.1242718 L16.7934656,21.5300971 L16.7934656,8 Z"/>
		</svg>',
			'bookmark'            => '<svg xmlns="http://www.w3.org/2000/svg" width="15" height="18" viewBox="0 0 15 18">
		<path d="M12.598889,2.699762 C12.598889,2.20275 12.195981,1.799841 11.698969,1.799841 L2.699762,1.799841 C2.20275,1.799841 1.799841,2.20275 1.799841,2.699762 L1.799841,15.349777 L6.676297,11.866594 C6.989197,11.643094 7.409533,11.643094 7.722434,11.866594 L12.598889,15.349777 L12.598889,2.699762 Z M1.422989,17.830788 C0.82736,18.256238 0,17.830464 0,17.098493 L0,2.699762 C0,1.208725 1.208725,0 2.699762,0 L11.698969,0 C13.190006,0 14.398731,1.208725 14.398731,2.699762 L14.398731,17.098493 C14.398731,17.830464 13.571371,18.256238 12.975742,17.830788 L7.199365,13.704805 L1.422989,17.830788 Z"/>
		</svg>',
			'calendar'            => '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="19" viewBox="0 0 18 19">
		<path fill="#1A1A1B" d="M4.60069444,4.09375 L3.25,4.09375 C2.47334957,4.09375 1.84375,4.72334957 1.84375,5.5 L1.84375,7.26736111 L16.15625,7.26736111 L16.15625,5.5 C16.15625,4.72334957 15.5266504,4.09375 14.75,4.09375 L13.3993056,4.09375 L13.3993056,4.55555556 C13.3993056,5.02154581 13.0215458,5.39930556 12.5555556,5.39930556 C12.0895653,5.39930556 11.7118056,5.02154581 11.7118056,4.55555556 L11.7118056,4.09375 L6.28819444,4.09375 L6.28819444,4.55555556 C6.28819444,5.02154581 5.9104347,5.39930556 5.44444444,5.39930556 C4.97845419,5.39930556 4.60069444,5.02154581 4.60069444,4.55555556 L4.60069444,4.09375 Z M6.28819444,2.40625 L11.7118056,2.40625 L11.7118056,1 C11.7118056,0.534009742 12.0895653,0.15625 12.5555556,0.15625 C13.0215458,0.15625 13.3993056,0.534009742 13.3993056,1 L13.3993056,2.40625 L14.75,2.40625 C16.4586309,2.40625 17.84375,3.79136906 17.84375,5.5 L17.84375,15.875 C17.84375,17.5836309 16.4586309,18.96875 14.75,18.96875 L3.25,18.96875 C1.54136906,18.96875 0.15625,17.5836309 0.15625,15.875 L0.15625,5.5 C0.15625,3.79136906 1.54136906,2.40625 3.25,2.40625 L4.60069444,2.40625 L4.60069444,1 C4.60069444,0.534009742 4.97845419,0.15625 5.44444444,0.15625 C5.9104347,0.15625 6.28819444,0.534009742 6.28819444,1 L6.28819444,2.40625 Z M1.84375,8.95486111 L1.84375,15.875 C1.84375,16.6516504 2.47334957,17.28125 3.25,17.28125 L14.75,17.28125 C15.5266504,17.28125 16.15625,16.6516504 16.15625,15.875 L16.15625,8.95486111 L1.84375,8.95486111 Z"/>
		</svg>',
			'chevron-down'        => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="12" viewBox="0 0 20 12">
		<polygon fill="#1A1A1B" fill-rule="evenodd" points="1319.899 365.778 1327.678 358 1329.799 360.121 1319.899 370.021 1310 360.121 1312.121 358" transform="translate(-1310 -358)"/>
		</svg>',
			'comment'             => '<svg xmlns="http://www.w3.org/2000/svg" width="19" height="19" viewBox="0 0 19 19">
		<path d="M9.43016863,13.2235931 C9.58624731,13.094699 9.7823475,13.0241935 9.98476849,13.0241935 L15.0564516,13.0241935 C15.8581553,13.0241935 16.5080645,12.3742843 16.5080645,11.5725806 L16.5080645,3.44354839 C16.5080645,2.64184472 15.8581553,1.99193548 15.0564516,1.99193548 L3.44354839,1.99193548 C2.64184472,1.99193548 1.99193548,2.64184472 1.99193548,3.44354839 L1.99193548,11.5725806 C1.99193548,12.3742843 2.64184472,13.0241935 3.44354839,13.0241935 L5.76612903,13.0241935 C6.24715123,13.0241935 6.63709677,13.4141391 6.63709677,13.8951613 L6.63709677,15.5301903 L9.43016863,13.2235931 Z M3.44354839,14.766129 C1.67980032,14.766129 0.25,13.3363287 0.25,11.5725806 L0.25,3.44354839 C0.25,1.67980032 1.67980032,0.25 3.44354839,0.25 L15.0564516,0.25 C16.8201997,0.25 18.25,1.67980032 18.25,3.44354839 L18.25,11.5725806 C18.25,13.3363287 16.8201997,14.766129 15.0564516,14.766129 L10.2979143,14.766129 L6.32072889,18.0506004 C5.75274472,18.5196577 4.89516129,18.1156602 4.89516129,17.3790323 L4.89516129,14.766129 L3.44354839,14.766129 Z"/>
		</svg>',
			'cross'               => '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 16 16">
		<polygon fill="#1A1A1B" fill-rule="evenodd" points="6.852 7.649 .399 1.195 1.445 .149 7.899 6.602 14.352 .149 15.399 1.195 8.945 7.649 15.399 14.102 14.352 15.149 7.899 8.695 1.445 15.149 .399 14.102"/>
		</svg>',
			'ellipsis'            => '<svg xmlns="http://www.w3.org/2000/svg" width="26" height="7" viewBox="0 0 26 7">
		<path fill-rule="evenodd" d="M332.5,45 C330.567003,45 329,43.4329966 329,41.5 C329,39.5670034 330.567003,38 332.5,38 C334.432997,38 336,39.5670034 336,41.5 C336,43.4329966 334.432997,45 332.5,45 Z M342,45 C340.067003,45 338.5,43.4329966 338.5,41.5 C338.5,39.5670034 340.067003,38 342,38 C343.932997,38 345.5,39.5670034 345.5,41.5 C345.5,43.4329966 343.932997,45 342,45 Z M351.5,45 C349.567003,45 348,43.4329966 348,41.5 C348,39.5670034 349.567003,38 351.5,38 C353.432997,38 355,39.5670034 355,41.5 C355,43.4329966 353.432997,45 351.5,45 Z" transform="translate(-329 -38)"/>
		</svg>',
			'edit'                => '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 18 18">
		<path fill="#1A1A1B" d="M14.7272727,11.1763636 C14.7272727,10.7244943 15.0935852,10.3581818 15.5454545,10.3581818 C15.9973239,10.3581818 16.3636364,10.7244943 16.3636364,11.1763636 L16.3636364,15.5454545 C16.3636364,16.9010626 15.2646989,18 13.9090909,18 L2.45454545,18 C1.09893743,18 0,16.9010626 0,15.5454545 L0,4.09090909 C0,2.73530107 1.09893743,1.63636364 2.45454545,1.63636364 L6.82363636,1.63636364 C7.2755057,1.63636364 7.64181818,2.00267611 7.64181818,2.45454545 C7.64181818,2.9064148 7.2755057,3.27272727 6.82363636,3.27272727 L2.45454545,3.27272727 C2.00267611,3.27272727 1.63636364,3.63903975 1.63636364,4.09090909 L1.63636364,15.5454545 C1.63636364,15.9973239 2.00267611,16.3636364 2.45454545,16.3636364 L13.9090909,16.3636364 C14.3609602,16.3636364 14.7272727,15.9973239 14.7272727,15.5454545 L14.7272727,11.1763636 Z M6.54545455,9.33890201 L6.54545455,11.4545455 L8.66109799,11.4545455 L16.0247344,4.09090909 L13.9090909,1.97526564 L6.54545455,9.33890201 Z M14.4876328,0.239639906 L17.7603601,3.51236718 C18.07988,3.83188705 18.07988,4.34993113 17.7603601,4.669451 L9.57854191,12.8512692 C9.42510306,13.004708 9.21699531,13.0909091 9,13.0909091 L5.72727273,13.0909091 C5.27540339,13.0909091 4.90909091,12.7245966 4.90909091,12.2727273 L4.90909091,9 C4.90909091,8.78300469 4.99529196,8.57489694 5.14873082,8.42145809 L13.330549,0.239639906 C13.6500689,-0.0798799688 14.1681129,-0.0798799688 14.4876328,0.239639906 Z"/>
		</svg>',
			'folder'              => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="19" viewBox="0 0 20 19">
		<path fill="#1A1A1B" d="M2.8,1.85 C2.275329,1.85 1.85,2.27532949 1.85,2.8 L1.85,15.4 C1.85,15.9246705 2.275329,16.35 2.8,16.35 L17.2,16.35 C17.724671,16.35 18.15,15.9246705 18.15,15.4 L18.15,5.5 C18.15,4.97532949 17.724671,4.55 17.2,4.55 L9.1,4.55 C8.8158,4.55 8.550403,4.40796403 8.392757,4.17149517 L6.845094,1.85 L2.8,1.85 Z M17.2,2.85 C18.663555,2.85 19.85,4.03644541 19.85,5.5 L19.85,15.4 C19.85,16.8635546 18.663555,18.05 17.2,18.05 L2.8,18.05 C1.336445,18.05 0.15,16.8635546 0.15,15.4 L0.15,2.8 C0.15,1.33644541 1.336445,0.15 2.8,0.15 L7.3,0.15 C7.5842,0.15 7.849597,0.292035965 8.007243,0.528504833 L9.554906,2.85 L17.2,2.85 Z"/>
		</svg>',
			'link'                => '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 18 18">
		<path d="M6.70846497,10.3082552 C6.43780491,9.94641406 6.5117218,9.43367048 6.87356298,9.16301045 C7.23540415,8.89235035 7.74814771,8.96626726 8.01880776,9.32810842 C8.5875786,10.0884893 9.45856383,10.5643487 10.4057058,10.6321812 C11.3528479,10.7000136 12.2827563,10.3531306 12.9541853,9.68145807 L15.3987642,7.23705399 C16.6390369,5.9529049 16.6212992,3.91168563 15.3588977,2.6492841 C14.0964962,1.38688258 12.0552769,1.36914494 10.77958,2.60113525 L9.37230725,4.00022615 C9.05185726,4.31881314 8.53381538,4.31730281 8.21522839,3.99685275 C7.89664141,3.67640269 7.89815174,3.15836082 8.21860184,2.83977385 L9.63432671,1.43240056 C11.5605503,-0.42800847 14.6223793,-0.401402004 16.5159816,1.49220028 C18.4095838,3.38580256 18.4361903,6.44763148 16.5658147,8.38399647 L14.1113741,10.838437 C13.1043877,11.8457885 11.7095252,12.366113 10.2888121,12.2643643 C8.86809903,12.1626156 7.56162126,11.4488264 6.70846497,10.3082552 Z M11.291535,7.6917448 C11.5621951,8.05358597 11.4882782,8.56632952 11.126437,8.83698955 C10.7645959,9.10764965 10.2518523,9.03373274 9.98119227,8.67189158 C9.4124214,7.91151075 8.54143617,7.43565129 7.59429414,7.36781884 C6.6471521,7.29998638 5.71724372,7.64686937 5.04581464,8.31854193 L2.60123581,10.762946 C1.36096312,12.0470951 1.37870076,14.0883144 2.64110228,15.3507159 C3.90350381,16.6131174 5.94472309,16.630855 7.21873082,15.400549 L8.61782171,14.0014581 C8.93734159,13.6819382 9.45538568,13.6819382 9.77490556,14.0014581 C10.0944254,14.320978 10.0944254,14.839022 9.77490556,15.1585419 L8.36567329,16.5675994 C6.43944966,18.4280085 3.37762074,18.401402 1.48401846,16.5077998 C-0.409583822,14.6141975 -0.436190288,11.5523685 1.43418536,9.61600353 L3.88862594,7.16156298 C4.89561225,6.15421151 6.29047483,5.63388702 7.71118789,5.7356357 C9.13190097,5.83738438 10.4383788,6.55117356 11.291535,7.6917448 Z"/>
		</svg>',
			'search'              => '<svg xmlns="http://www.w3.org/2000/svg" width="23" height="23" viewBox="0 0 23 23">
		<path d="M38.710696,48.0601792 L43,52.3494831 L41.3494831,54 L37.0601792,49.710696 C35.2632422,51.1481185 32.9839107,52.0076499 30.5038249,52.0076499 C24.7027226,52.0076499 20,47.3049272 20,41.5038249 C20,35.7027226 24.7027226,31 30.5038249,31 C36.3049272,31 41.0076499,35.7027226 41.0076499,41.5038249 C41.0076499,43.9839107 40.1481185,46.2632422 38.710696,48.0601792 Z M36.3875844,47.1716785 C37.8030221,45.7026647 38.6734666,43.7048964 38.6734666,41.5038249 C38.6734666,36.9918565 35.0157934,33.3341833 30.5038249,33.3341833 C25.9918565,33.3341833 22.3341833,36.9918565 22.3341833,41.5038249 C22.3341833,46.0157934 25.9918565,49.6734666 30.5038249,49.6734666 C32.7048964,49.6734666 34.7026647,48.8030221 36.1716785,47.3875844 C36.2023931,47.347638 36.2360451,47.3092237 36.2726343,47.2726343 C36.3092237,47.2360451 36.347638,47.2023931 36.3875844,47.1716785 Z" transform="translate(-20 -31)"/>
		</svg>',
			'tag'                 => '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 18 18">
		<path fill="#1A1A1B" d="M15.4496399,8.42490555 L8.66109799,1.63636364 L1.63636364,1.63636364 L1.63636364,8.66081885 L8.42522727,15.44178 C8.57869221,15.5954158 8.78693789,15.6817418 9.00409091,15.6817418 C9.22124393,15.6817418 9.42948961,15.5954158 9.58327627,15.4414581 L15.4486339,9.57610048 C15.7651495,9.25692435 15.7649133,8.74206554 15.4496399,8.42490555 Z M16.6084423,10.7304545 L10.7406818,16.59822 C10.280287,17.0591273 9.65554997,17.3181054 9.00409091,17.3181054 C8.35263185,17.3181054 7.72789481,17.0591273 7.26815877,16.5988788 L0.239976954,9.57887876 C0.0863319284,9.4254126 0,9.21716044 0,9 L0,0.818181818 C0,0.366312477 0.366312477,0 0.818181818,0 L9,0 C9.21699531,0 9.42510306,0.0862010512 9.57854191,0.239639906 L16.6084423,7.26954545 C17.5601275,8.22691012 17.5601275,9.77308988 16.6084423,10.7304545 Z M5,6 C4.44771525,6 4,5.55228475 4,5 C4,4.44771525 4.44771525,4 5,4 C5.55228475,4 6,4.44771525 6,5 C6,5.55228475 5.55228475,6 5,6 Z"/>
		</svg>',
			'user'                => '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="20" viewBox="0 0 18 20">
		<path fill="#1A1A1B" d="M18,19 C18,19.5522847 17.5522847,20 17,20 C16.4477153,20 16,19.5522847 16,19 L16,17 C16,15.3431458 14.6568542,14 13,14 L5,14 C3.34314575,14 2,15.3431458 2,17 L2,19 C2,19.5522847 1.55228475,20 1,20 C0.44771525,20 0,19.5522847 0,19 L0,17 C0,14.2385763 2.23857625,12 5,12 L13,12 C15.7614237,12 18,14.2385763 18,17 L18,19 Z M9,10 C6.23857625,10 4,7.76142375 4,5 C4,2.23857625 6.23857625,0 9,0 C11.7614237,0 14,2.23857625 14,5 C14,7.76142375 11.7614237,10 9,10 Z M9,8 C10.6568542,8 12,6.65685425 12,5 C12,3.34314575 10.6568542,2 9,2 C7.34314575,2 6,3.34314575 6,5 C6,6.65685425 7.34314575,8 9,8 Z"/>
		</svg>',
		);

		/**
		 * Social Icons – domain mappings.
		 *
		 * By default, each Icon ID is matched against a .com TLD. To override this behavior,
		 * specify all the domains it covers (including the .com TLD too, if applicable).
		 *
		 * @var array
		 */
		public static $social_icons_map = array(
			'amazon'      => array(
				'amazon.com',
				'amazon.cn',
				'amazon.in',
				'amazon.fr',
				'amazon.de',
				'amazon.it',
				'amazon.nl',
				'amazon.es',
				'amazon.co',
				'amazon.ca',
			),
			'apple'       => array(
				'apple.com',
				'itunes.com',
			),
			'behance'     => array(
				'behance.net',
			),
			'codepen'     => array(
				'codepen.io',
			),
			'facebook'    => array(
				'facebook.com',
				'fb.me',
			),
			'feed'        => array(
				'feed',
			),
			'google-plus' => array(
				'plus.google.com',
			),
			'lastfm'      => array(
				'last.fm',
			),
			'mail'        => array(
				'mailto:',
			),
			'slideshare'  => array(
				'slideshare.net',
				'slideshare.com',
			),
			'pocket'      => array(
				'getpocket.com',
			),
			'twitch'      => array(
				'twitch.tv',
			),
			'wordpress'   => array(
				'wordpress.com',
				'wordpress.org',
			),
		);

		/**
		 * Social Icons – svg sources.
		 *
		 * @var array
		 */
		public static $social_icons = array(
			'500px'       => '<svg width="14" height="18" viewBox="0 0 14 18" xmlns="http://www.w3.org/2000/svg"><path d="M2.448 11.848c-.047-.152 0-.26.14-.325.141-.065.253-.109.335-.132.14-.047.263-.07.369-.07.105 0 .176.058.21.175.118.317.264.624.44.923s.358.554.545.765V9.756c0-.422.088-.835.264-1.24.176-.403.427-.764.756-1.08.328-.34.714-.6 1.16-.782a3.599 3.599 0 011.37-.273c.493 0 .956.094 1.39.281.433.188.811.443 1.133.765.323.322.578.7.765 1.134s.281.89.281 1.37c0 .493-.093.956-.281 1.39s-.442.811-.765 1.133c-.322.323-.7.578-1.133.765s-.897.281-1.39.281c-.198 0-.38-.008-.544-.026a3.053 3.053 0 01-.58-.132.64.64 0 01-.132-.132c-.064-.076-.061-.237.009-.483.012-.059.05-.155.114-.29.065-.135.167-.185.308-.15.011 0 .111.024.299.07.187.048.345.071.474.071.692 0 1.28-.24 1.767-.72.486-.481.73-1.067.73-1.758 0-.317-.062-.624-.185-.923s-.302-.56-.536-.783c-.235-.234-.504-.413-.81-.536s-.626-.184-.966-.184c-.34 0-.662.07-.967.21-.304.141-.574.34-.808.598-.211.223-.372.475-.484.756s-.167.563-.167.844v4.166c.364.21.75.375 1.16.492a4.783 4.783 0 003.12-.193 4.966 4.966 0 001.574-1.037 4.993 4.993 0 001.037-1.565c.246-.586.37-1.201.37-1.845a4.774 4.774 0 00-1.407-3.393A4.758 4.758 0 009.857 5.51a4.798 4.798 0 00-1.854-.36c-.645 0-1.263.12-1.855.36-.592.24-1.11.589-1.556 1.046l-.325.325c-.217.217-.366.396-.448.536l-.018.018c-.07.082-.149.152-.237.21-.088.06-.255.06-.5 0a.925.925 0 01-.361-.149c-.112-.076-.167-.167-.167-.272V.405c0-.094.038-.182.114-.264a.379.379 0 01.29-.123L11.712 0c.152 0 .243.076.272.229s.044.263.044.334-.012.179-.035.325c-.024.146-.111.22-.264.22H3.573v4.886c.281-.293.633-.58 1.055-.86s.773-.481 1.054-.599c.364-.152.742-.27 1.134-.351a5.786 5.786 0 012.364 0c.387.082.756.2 1.108.351.351.141.685.32 1.002.537.316.216.61.46.879.73a5.982 5.982 0 011.265 1.862 5.843 5.843 0 01.457 2.285c0 .399-.038.791-.114 1.178-.076.387-.19.756-.343 1.108s-.334.685-.545 1.002a5.754 5.754 0 01-1.6 1.6 5.45 5.45 0 01-1.001.526c-.352.165-.72.285-1.108.36a6.08 6.08 0 01-2.337 0 5.137 5.137 0 01-1.125-.36 5.198 5.198 0 01-.994-.527 6.12 6.12 0 01-1.6-1.6 5.154 5.154 0 01-.623-1.23 1.052 1.052 0 01-.053-.158zm4.201-.984c0-.024.006-.05.018-.08s.04-.061.088-.096l.65-.668-.668-.668c-.059-.047-.076-.109-.053-.185a.62.62 0 01.194-.255.371.371 0 01.15-.14.369.369 0 01.149-.035c.035 0 .061.003.079.008.017.006.032.021.044.044l.668.668.65-.65a.2.2 0 01.21-.053.803.803 0 01.413.387c.042.082.027.158-.043.228l-.65.668.632.633c.024.035.05.091.08.167.029.076-.021.185-.15.325a.778.778 0 01-.167.132.339.339 0 01-.167.044l-.105-.035a.17.17 0 01-.088-.07l-.633-.615c-.023.023-.132.134-.325.334-.194.199-.308.31-.343.334 0 .011-.012.023-.035.035s-.059.017-.106.017a.36.36 0 01-.158-.044.605.605 0 01-.158-.114v-.018c-.035-.035-.073-.076-.114-.123s-.062-.105-.062-.175zm1.424-8.912c.55 0 1.093.052 1.626.158s1.052.258 1.556.457c.457.199.879.425 1.265.677.387.251.75.536 1.09.852.094.094.12.188.08.281a1.168 1.168 0 01-.203.3c-.07.07-.138.134-.202.192a.298.298 0 01-.202.088.216.216 0 01-.08-.017.177.177 0 01-.078-.07 11.99 11.99 0 00-1.046-.792 5.607 5.607 0 00-1.029-.544 6.76 6.76 0 00-1.344-.422 7.2 7.2 0 00-1.433-.14c-.422 0-.858.043-1.31.13a7.633 7.633 0 00-1.31.379c-.14.047-.242.009-.307-.114a5.898 5.898 0 00-.15-.273 19.104 19.104 0 01-.06-.22.27.27 0 01-.01-.149c0-.047.015-.088.044-.123s.062-.065.097-.088a7.153 7.153 0 011.424-.413c.527-.1 1.055-.15 1.582-.15zm5.678 13.025c.175.164.26.29.255.377s-.039.15-.097.185l-.044.044-.026.026c-.364.364-.765.692-1.205.985s-.899.545-1.38.756c-.503.222-1.025.386-1.564.492a8.452 8.452 0 01-3.252 0 7.353 7.353 0 01-1.564-.492c-.48-.2-.938-.446-1.372-.739s-.837-.627-1.212-1.002a8.226 8.226 0 01-1.758-2.601 12.433 12.433 0 01-.317-.932 6.897 6.897 0 01-.21-.95c-.024-.14.035-.234.175-.28.14-.047.24-.077.299-.088.152-.024.281-.024.387 0s.164.093.175.21v.036c.024.094.068.284.132.571.065.287.179.63.343 1.028.164.41.375.806.633 1.187.258.38.557.741.896 1.081.329.328.683.618 1.064.87.38.252.782.472 1.204.66.434.187.885.327 1.353.421a7.154 7.154 0 002.83 0 6.72 6.72 0 001.354-.422 8.15 8.15 0 001.213-.668 6.895 6.895 0 001.055-.861l.052-.07a.26.26 0 01.123-.07.198.198 0 01.141.017.604.604 0 01.14.079c.06.041.118.09.176.15z"/></svg>',

			'amazon'      => '<svg width="18" height="16" viewBox="0 0 18 15" xmlns="http://www.w3.org/2000/svg"><path d="M16.004 11.571c.164-.07.29-.035.378.106.088.14.056.27-.097.386a8.912 8.912 0 01-1.635.95c-.586.281-1.199.516-1.838.703a13.92 13.92 0 01-3.842.563c-.867 0-1.706-.076-2.515-.229a13.056 13.056 0 01-4.484-1.74 12.657 12.657 0 01-1.899-1.442c-.082-.07-.094-.147-.035-.23.059-.081.14-.087.246-.017.633.376 1.301.704 2.005.985.703.293 1.43.542 2.18.747.75.206 1.524.361 2.321.466.786.106 1.583.159 2.392.159.551 0 1.114-.03 1.688-.088a17.495 17.495 0 001.723-.264 18.569 18.569 0 001.724-.44 17.07 17.07 0 001.688-.615zm1.934-1.09c.106.129.077.55-.088 1.266-.164.715-.51 1.29-1.037 1.723-.07.07-.138.091-.202.062s-.08-.085-.044-.167c.129-.305.278-.718.448-1.24.17-.522.196-.859.08-1.011-.13-.153-.464-.211-1.003-.176-.54.035-.967.07-1.284.105-.105 0-.164-.029-.176-.087-.011-.06.024-.124.106-.194.563-.387 1.21-.566 1.943-.536.733.03 1.152.114 1.257.255zM9.99 3.834v-.247c0-.21-.006-.43-.018-.659a1.181 1.181 0 00-.193-.607c-.106-.164-.252-.281-.44-.351s-.369-.106-.545-.106c-.34 0-.642.088-.905.264-.264.176-.431.44-.502.791a.48.48 0 01-.123.229.269.269 0 01-.193.088L5.348 3.06a.466.466 0 01-.185-.114c-.053-.053-.073-.132-.062-.238.2-1.043.695-1.755 1.486-2.136A5.592 5.592 0 019.04 0c.422 0 .885.067 1.39.202s.92.36 1.248.677c.422.41.657.874.703 1.39s.07 1.06.07 1.635v2.743c0 .41.074.73.22.958.147.23.297.449.449.66.059.094.09.182.097.264a.288.288 0 01-.097.228c-.2.141-.434.331-.703.572-.27.24-.493.436-.669.589a.278.278 0 01-.202.088.486.486 0 01-.22-.053c-.281-.246-.477-.445-.589-.598a9.661 9.661 0 01-.395-.598c-.493.493-.944.833-1.354 1.02-.41.188-.932.282-1.566.282-.738 0-1.362-.23-1.872-.686-.51-.457-.765-1.143-.765-2.058 0-.727.176-1.327.527-1.802A3.337 3.337 0 016.684 4.43a5.668 5.668 0 011.635-.422c.61-.07 1.167-.129 1.67-.176zM9.638 7.49c.176-.316.279-.624.308-.923a9.65 9.65 0 00.044-.94V5.24c-.633 0-1.231.109-1.794.326s-.844.694-.844 1.433c0 .387.1.7.3.94s.462.361.79.361c.247 0 .478-.073.695-.22.217-.146.384-.342.501-.589z"/></svg>',

			'apple'       => '<svg width="16" height="19" viewBox="0 0 16 19" xmlns="http://www.w3.org/2000/svg"><path d="M14.94 6.186c-.633.21-1.143.68-1.53 1.406-.386.727-.58 1.418-.58 2.074a4.7 4.7 0 00.669 2.409c.445.761 1.054 1.248 1.828 1.459-.106.363-.252.75-.44 1.16-.199.398-.422.785-.668 1.16s-.504.72-.773 1.037a5.975 5.975 0 01-.826.756 2.503 2.503 0 01-.739.343c-.27.076-.574.114-.914.114-.21 0-.442-.023-.694-.07s-.548-.14-.888-.282a4.333 4.333 0 00-.738-.263 2.533 2.533 0 00-.598-.088c-.14 0-.354.04-.641.123-.287.082-.636.205-1.046.37-.223.07-.428.128-.615.175-.188.047-.346.07-.475.07-.363 0-.709-.073-1.037-.22s-.662-.354-1.002-.624a9.465 9.465 0 01-1.389-1.582 10.162 10.162 0 01-1.01-1.837 10.446 10.446 0 01-.625-2.03C.07 11.143 0 10.428 0 9.702c0-.715.105-1.407.316-2.075a6.08 6.08 0 01.932-1.784c.398-.521.878-.94 1.441-1.257s1.19-.474 1.88-.474c.423 0 .792.04 1.108.123.317.082.616.193.897.334.258.152.507.26.747.325s.483.097.73.097c.187 0 .392-.018.615-.053.222-.035.439-.1.65-.193.539-.211.99-.37 1.353-.475.364-.105.65-.158.862-.158.527 0 1.005.079 1.432.237.428.158.818.396 1.17.712.14.14.278.305.412.492.135.188.267.399.396.633zM7.557 4.217c0-.492.097-.96.29-1.406.194-.445.46-.879.8-1.3.328-.364.715-.66 1.16-.888.446-.229.914-.39 1.406-.484a4.55 4.55 0 01-.254 1.494c-.17.492-.425.944-.765 1.354-.387.422-.794.732-1.222.931-.428.2-.9.3-1.415.3z"/></svg>',

			'bandcamp'    => '<svg width="19" height="6" viewBox="0 0 19 6" xmlns="http://www.w3.org/2000/svg"><path d="M.32 5.811L3.449.361h6.68l-3.13 5.45H.32zm13.289-3.445c.07.105.14.214.21.325.071.111.13.225.177.343a2.953 2.953 0 01.105.808c0 .14-.009.276-.026.405a2.422 2.422 0 01-.08.369c-.034.129-.078.246-.131.351-.053.106-.114.211-.185.317a2.01 2.01 0 01-.237.263 1.046 1.046 0 01-.29.194 2.079 2.079 0 01-.378.132c-.123.029-.255.043-.395.043a2.389 2.389 0 01-.422-.035c-.07 0-.141-.008-.211-.026s-.14-.044-.211-.08a1.46 1.46 0 01-.317-.175 1.12 1.12 0 01-.14-.167 1.724 1.724 0 01-.106-.184h-.035v.562h-.808V.361h.878v2.005a.73.73 0 01.106-.14l.105-.106c.059-.036.109-.07.15-.106.04-.035.085-.07.132-.105a6.737 6.737 0 01.351-.106.492.492 0 01.185-.035h.167c.164 0 .313.012.448.035.135.024.255.059.36.106.141.058.258.123.352.193.094.07.176.158.246.264zm-.457 1.968a.93.93 0 00.044-.228 2.46 2.46 0 01.026-.229c0-.093-.003-.184-.009-.272a.773.773 0 00-.061-.255.938.938 0 00-.097-.246c-.04-.07-.079-.14-.114-.21-.059-.06-.114-.112-.167-.16s-.114-.087-.185-.122c-.07-.059-.149-.097-.237-.114s-.173-.027-.255-.027c-.175 0-.34.035-.492.106a.853.853 0 00-.351.28 1.377 1.377 0 00-.211.45 2.007 2.007 0 00-.044.843l.044.255a.938.938 0 00.096.246c.041.07.08.14.115.21.058.06.117.112.175.159a.909.909 0 00.211.123.688.688 0 00.211.08c.082.017.164.026.246.026.094 0 .182-.012.264-.036s.158-.058.228-.105a.868.868 0 00.194-.132.972.972 0 00.158-.184 2.87 2.87 0 01.123-.22.616.616 0 00.088-.238zm3.973.668a1.1 1.1 0 00.22-.307 1.12 1.12 0 00.096-.396h.879c-.059.27-.135.501-.229.695a1.66 1.66 0 01-.369.5c-.164.141-.345.247-.545.317-.199.07-.427.106-.685.106-.164 0-.317-.018-.457-.053a3.004 3.004 0 01-.387-.123 2.1 2.1 0 01-.343-.194c-.1-.07-.184-.146-.255-.228a1.352 1.352 0 01-.228-.299 2.92 2.92 0 01-.158-.334c-.035-.14-.068-.278-.097-.413a2.045 2.045 0 010-.853c.03-.14.062-.27.097-.386.058-.14.117-.267.176-.378s.128-.22.21-.325c.07-.094.159-.176.264-.247.106-.07.217-.128.334-.175.129-.059.27-.103.422-.132a2.255 2.255 0 011.125.062c.105.035.205.073.299.114.094.04.181.085.263.132.07.07.141.143.211.22.07.075.13.155.176.237.035.093.07.193.106.298.035.106.058.217.07.334h-.879a.964.964 0 00-.114-.325.967.967 0 00-.202-.237c-.07-.059-.159-.103-.264-.132s-.217-.044-.334-.044a.701.701 0 00-.185.026c-.064.018-.12.033-.167.044-.07.024-.14.053-.21.088a.67.67 0 00-.317.317c-.035.07-.07.146-.105.228-.036.106-.062.217-.08.334s-.026.252-.026.404a3.348 3.348 0 00.035.493c.024.07.05.146.08.228a.74.74 0 00.096.194c.035.07.085.131.15.184.064.053.12.097.167.132.07.035.149.061.237.079a1.299 1.299 0 00.633-.026.819.819 0 00.29-.159z"/></svg>',

			'behance'     => '<svg width="19" height="12" viewBox="0 0 19 12" xmlns="http://www.w3.org/2000/svg"><path d="M16.52 2.396h-4.499V1.342h4.499v1.054zM9.069 6.649c.176.246.308.512.395.8.088.286.132.606.132.957 0 .281-.026.542-.079.782-.053.24-.132.46-.237.659-.117.21-.25.398-.396.562a3.975 3.975 0 01-.483.457 4.816 4.816 0 01-.597.317c-.211.093-.434.175-.668.246-.223.046-.457.087-.703.123a5.213 5.213 0 01-.738.052H.318V.427h5.201c.258 0 .51.02.756.062.246.04.486.09.72.15.211.046.405.113.58.201.176.088.346.185.51.29.152.106.29.235.413.387.123.152.231.316.325.492.082.152.144.34.185.562.04.223.061.469.061.738 0 .258-.03.507-.088.747-.058.24-.158.448-.299.624-.117.188-.272.36-.465.518-.194.159-.413.29-.66.396.352.105.65.246.897.422.246.175.451.386.615.632zM2.672 4.927h2.53c.223 0 .425-.015.607-.044.182-.03.343-.097.483-.203.14-.105.246-.234.317-.386.07-.152.105-.352.105-.598a2.03 2.03 0 00-.026-.334.81.81 0 00-.114-.298c-.047-.07-.1-.138-.159-.202s-.134-.126-.228-.185a9.83 9.83 0 00-.237-.114 1.404 1.404 0 00-.29-.097 5.861 5.861 0 01-.317-.044 1.921 1.921 0 00-.316-.026H2.672v2.53zm4.499 3.338c0-.28-.041-.518-.123-.711s-.205-.355-.369-.484a1.573 1.573 0 00-.545-.263 2.697 2.697 0 00-.72-.088H2.672v3.058h2.671a3.8 3.8 0 00.36-.018c.124-.012.238-.03.343-.053.106-.035.202-.07.29-.105.088-.035.179-.082.273-.14a.959.959 0 00.228-.185.741.741 0 00.158-.273c.047-.082.088-.184.123-.307s.053-.267.053-.43zm11.141-.105h-5.8c0 .281.05.568.15.861s.237.521.413.685c.152.153.349.273.589.36.24.089.512.132.817.132.223 0 .43-.032.624-.096.193-.065.372-.161.536-.29.152-.106.275-.211.369-.316s.17-.246.228-.422h1.933c-.152.539-.351.972-.597 1.3a3.439 3.439 0 01-.844.809c-.316.21-.667.374-1.054.492s-.808.175-1.265.175a4.577 4.577 0 01-1.722-.316 4.25 4.25 0 01-.686-.369 3.793 3.793 0 01-.58-.475 3.888 3.888 0 01-.474-.623 4.664 4.664 0 01-.37-.747 5.992 5.992 0 01-.201-.791 4.86 4.86 0 010-1.757c.052-.282.131-.557.237-.826.082-.258.193-.495.334-.712a5.31 5.31 0 01.474-.624 4.11 4.11 0 01.607-.5c.216-.147.448-.285.694-.414.257-.105.527-.184.808-.237a4.76 4.76 0 011.819.018c.299.064.583.16.852.29.246.14.472.298.677.474.205.176.39.37.553.58.176.223.32.466.43.73a5.57 5.57 0 01.423 1.721c.029.323.038.618.026.888zm-2.214-1.441c-.035-.258-.094-.492-.176-.703s-.187-.387-.316-.527a1.301 1.301 0 00-.536-.343 2.19 2.19 0 00-.73-.114 1.96 1.96 0 00-.474.053c-.14.035-.275.088-.404.158a1.803 1.803 0 00-.308.176c-.1.07-.184.152-.255.246-.07.082-.125.17-.166.263-.041.094-.08.193-.115.299a1.135 1.135 0 00-.105.492h3.585z"/></svg>',

			'codepen'     => '<svg width="18" height="19" viewBox="0 0 18 19" xmlns="http://www.w3.org/2000/svg"><path d="M18 6.38v5.5c0 .012-.003.03-.009.053a.29.29 0 00-.009.07v.054a.873.873 0 01-.044.132 2.497 2.497 0 01-.061.132c-.012 0-.018.003-.018.01v.008a.078.078 0 01-.009.035c-.005.012-.014.018-.026.018a.235.235 0 01-.044.061l-.026.027c-.024.012-.041.026-.053.044s-.024.032-.035.044c-.012 0-.02.003-.027.009l-.026.026-8.147 5.413a.135.135 0 01-.052.017c-.012 0-.024.006-.036.018-.023.012-.04.018-.052.018s-.03.005-.053.017c-.035.012-.08.024-.132.035a.69.69 0 01-.3 0 1.357 1.357 0 01-.131-.035.122.122 0 01-.053-.009l-.053-.026a.078.078 0 01-.035-.009l-.053-.026-8.129-5.413a.77.77 0 01-.123-.114 2.032 2.032 0 00-.106-.115H.15c-.006 0-.01-.006-.01-.017-.01 0-.017-.003-.017-.01v-.008a2.499 2.499 0 01-.061-.132.873.873 0 01-.044-.133v-.052c0-.024-.003-.047-.01-.071S0 11.91 0 11.898V6.38c0-.012.003-.03.009-.053s.009-.047.009-.07v-.053a.873.873 0 01.044-.132c.017-.042.038-.086.061-.133.012 0 .018-.003.018-.009v-.008l.017-.018a1.15 1.15 0 01.23-.23L8.515.264C8.54.25 8.557.245 8.57.245s.023-.006.035-.018A.135.135 0 018.657.21c.012 0 .03-.006.053-.018a.819.819 0 01.58 0c.024 0 .041.003.053.009l.053.026c.012 0 .023.003.035.01l.053.026 8.129 5.412a.77.77 0 01.123.114c.035.042.07.08.106.115h.008c.006 0 .01.006.01.018.011 0 .017.003.017.008v.01c.023.046.044.09.061.132a.873.873 0 01.044.132v.053c0 .023.003.047.01.07s.008.041.008.053zm-9.009 4.46l2.551-1.71-2.55-1.71L6.44 9.13l2.551 1.71zm.845-4.918l3.22 2.168 2.551-1.728-5.771-3.825v3.385zm-1.69 0V2.537l-5.77 3.825 2.55 1.728 3.22-2.168zM3.397 9.13L1.689 7.967v2.31L3.396 9.13zm1.53 1.023l-2.55 1.727 5.77 3.826v-3.385l-3.22-2.168zm4.91 2.168v3.385l5.771-3.826-2.551-1.727-3.22 2.168zm4.75-3.19l1.707 1.163v-2.31L14.587 9.13z"/></svg>',

			'deviantart'  => '<svg width="12" height="19" viewBox="0 0 12 19" xmlns="http://www.w3.org/2000/svg"><path d="M11.48.139h-.01v3.608L8.32 9.753l.29.284h2.86v4.513H6.14l-.46.249-1.58 3.003-.32.337H.68v-3.607l3.15-6.006-.29-.284H.68V3.747h5.35l.47-.267L8.08.477 8.4.139z"/></svg>',

			'digg'        => '<svg width="18" height="11" viewBox="0 0 18 11" xmlns="http://www.w3.org/2000/svg"><path d="M5.41 2.554h1.793v6.117H5.41V2.554zM2.879.023h1.793V8.67H-.004V2.554h2.883V.023zm0 3.973h-1.09V7.23h1.09V3.996zM5.41.023h1.793V1.85H5.41V.023zm2.496 2.531h4.676v8.297H7.906V9.41h2.883V8.67H7.906V2.554zM9.7 7.23h1.09V3.995h-1.09V7.23zm8.297-4.676v8.297H13.32V9.41h2.883V8.67H13.32V2.554h4.676zm-1.793 1.441h-1.09V7.23h1.09V3.995z"/></svg>',

			'dribbble'    => '<svg width="19" height="19" viewBox="0 0 19 19" xmlns="http://www.w3.org/2000/svg"><path d="M9.68.139c1.254 0 2.426.234 3.516.703 1.09.48 2.042 1.128 2.856 1.943s1.456 1.766 1.925 2.856.703 2.256.703 3.498c0 1.242-.234 2.408-.703 3.498s-1.11 2.042-1.925 2.856-1.767 1.462-2.856 1.943c-1.102.468-2.274.703-3.516.703s-2.408-.235-3.498-.703c-1.09-.48-2.042-1.128-2.856-1.943s-1.462-1.766-1.943-2.856A8.758 8.758 0 01.68 9.139c0-1.242.234-2.408.703-3.498.48-1.09 1.128-2.042 1.943-2.856S5.092 1.323 6.182.842A8.758 8.758 0 019.68.14zm5.977 4.148a7.736 7.736 0 01-1.16 1.16 12.69 12.69 0 01-1.301.932c-.516.328-1.125.639-1.828.932.093.176.181.36.263.554.082.193.158.383.229.57.035.06.064.121.088.185.023.065.052.132.088.203a12.923 12.923 0 011.88-.106c.622.012 1.178.041 1.67.088s.897.1 1.213.158c.328.059.522.094.58.106 0-.914-.155-1.779-.466-2.593a7.84 7.84 0 00-1.256-2.189zM9.68 1.44a7.741 7.741 0 00-1.793.21c.059.083.182.253.37.51.187.27.412.604.676 1.002.264.399.56.856.888 1.372.316.503.632 1.043.949 1.617 1.371-.516 2.37-1.081 2.997-1.697.627-.615.964-.975 1.01-1.08a7.29 7.29 0 00-1.09-.81 8.176 8.176 0 00-1.22-.606 6.813 6.813 0 00-1.346-.378 7.332 7.332 0 00-1.44-.14zm-3.234.738a7.587 7.587 0 00-1.512.932 8.169 8.169 0 00-1.257 1.248 7.56 7.56 0 00-.94 1.494 7.7 7.7 0 00-.58 1.705c.093 0 .345-.006.755-.018.41 0 .938-.026 1.582-.079a30.18 30.18 0 002.198-.272c.832-.13 1.699-.31 2.601-.545a45.446 45.446 0 00-.966-1.617 92.45 92.45 0 00-1.881-2.848zm-4.43 6.96c0 .493.047.974.14 1.442a7.136 7.136 0 001.002 2.584c.247.399.522.768.827 1.108l.492-.739c.281-.41.685-.884 1.213-1.423a14.583 14.583 0 011.916-1.618c.761-.539 1.64-.966 2.636-1.283l.15-.061a.938.938 0 01.132-.044l-.29-.65c-.1-.223-.203-.44-.308-.651-.973.293-1.916.516-2.83.668s-1.734.26-2.461.325c-.727.065-1.324.097-1.793.097-.469.012-.744.012-.826 0v.246zm7.664 7.7c.55 0 1.078-.059 1.582-.176.504-.117.984-.27 1.441-.457a11.367 11.367 0 00-.158-.79c-.07-.376-.17-.818-.299-1.328a31.427 31.427 0 00-.492-1.714c-.199-.633-.433-1.3-.703-2.004 0 .024-.006.036-.018.036h-.017c-1.09.375-2.01.826-2.76 1.353-.762.516-1.38 1.017-1.854 1.503-.475.486-.824.917-1.046 1.292a8.306 8.306 0 00-.387.668c.656.492 1.386.885 2.188 1.178s1.644.439 2.523.439zm4.324-1.336a8.092 8.092 0 001.196-.984c.363-.375.685-.78.966-1.213s.516-.897.703-1.389.323-1.008.405-1.547a8.474 8.474 0 00-.51-.158 11.686 11.686 0 00-2.531-.316c-.551-.012-1.131.029-1.74.123.257.691.474 1.347.65 1.969.176.609.325 1.157.448 1.643s.214.894.273 1.222c.07.316.117.533.14.65z"/></svg>',

			'etsy'        => '<svg width="16" height="19" viewBox="0 0 16 19" xmlns="http://www.w3.org/2000/svg"><path d="M.83.841A.423.423 0 01.9.543C.96.46 1.076.42 1.251.42c.082 0 .333.006.754.017.422.012.924.024 1.51.036s1.21.023 1.876.035 1.287.017 1.86.017c2.35 0 3.947-.064 4.79-.193.842-.128 1.467-.193 1.877-.193.21.012.35.082.42.21s.106.246.106.352-.035.26-.105.465a2.25 2.25 0 00-.105.728c0 .28.023.558.07.833.046.275.07.547.07.816 0 .245-.08.412-.237.5a.901.901 0 01-.43.131c-.105 0-.19-.029-.254-.087s-.097-.193-.097-.404c-.011-.398-.16-.824-.447-1.28-.287-.457-1.868-.685-4.746-.685-1.45 0-2.31.097-2.579.29s-.403.517-.403.973v4.456c0 .246.038.413.114.5.076.088.213.132.412.132h4.176c.397 0 .775-.117 1.131-.35s.565-.644.623-1.229c.012-.292.056-.47.132-.535s.196-.097.36-.097c.151 0 .28.065.385.193.106.13.152.334.14.614-.034.293-.072.597-.113.913-.041.316-.062.678-.062 1.087 0 .422.03.825.088 1.211.058.386.088.731.088 1.035 0 .292-.14.46-.421.5s-.457-.102-.527-.43c-.07-.292-.228-.646-.473-1.061-.246-.415-.825-.623-1.737-.623h-3.72c-.21 0-.362.067-.456.202-.093.134-.14.371-.14.71v4.737c0 .421.108.813.325 1.176s.74.543 1.57.543c.42 0 .912.006 1.473.018a20.7 20.7 0 001.685-.07 10.113 10.113 0 001.631-.263c.527-.13.965-.328 1.316-.597.28-.222.485-.517.614-.886.129-.368.24-.833.333-1.395.047-.245.26-.339.64-.28.38.058.553.386.518.982-.011.28-.108.67-.29 1.167-.18.497-.271.938-.271 1.324v.957c0 .345-.14.517-.421.517-.246 0-.445-.047-.597-.14-.152-.094-.772-.14-1.86-.14-.537 0-1.35.005-2.438.017-1.088 0-2.207.006-3.36.018-1.152.011-2.213.017-3.184.017-.97.012-1.614.018-1.93.018-.222 0-.37-.035-.447-.106s-.114-.175-.114-.315.044-.258.132-.351.23-.14.43-.14c.187 0 .579-.088 1.175-.264.596-.175.895-.79.895-1.842V3.858c0-.561-.1-1.088-.299-1.579-.198-.491-.649-.772-1.35-.842-.281-.035-.439-.102-.474-.202S.83 1.005.83.841z"/></svg>',

			'facebook'    => '<svg width="11" height="19" viewBox="0 0 11 19" xmlns="http://www.w3.org/2000/svg"><path d="M7.05 18.437H3.745V10.21H.968V7.011h2.777V4.656c0-.692.1-1.295.3-1.81.198-.528.477-.97.834-1.328a3.367 3.367 0 011.292-.8A4.903 4.903 0 017.86.437c.586 0 1.107.015 1.564.044.457.03.756.062.897.097v2.847H8.632c-.668 0-1.099.144-1.292.431-.193.287-.29.66-.29 1.116v2.04h3.164l-.422 3.199H7.05v8.226z"/></svg>',

			'feed'        => '<svg width="19" height="19" viewBox="0 0 19 19" xmlns="http://www.w3.org/2000/svg"><path d="M5.12 15.714c0 .668-.232 1.233-.697 1.696a2.321 2.321 0 01-1.703.695c-.67 0-1.238-.232-1.703-.695a2.303 2.303 0 01-.697-1.696c0-.645.232-1.204.697-1.679a2.292 2.292 0 011.703-.712c.67 0 1.238.235 1.703.704.465.468.697 1.03.697 1.687zM.32 9.702V6.257c1.647 0 3.188.31 4.624.932 1.447.632 2.708 1.485 3.785 2.557s1.926 2.33 2.55 3.77a11.6 11.6 0 01.935 4.624H8.755a8.552 8.552 0 00-.635-3.243 8.383 8.383 0 00-1.835-2.734A8.436 8.436 0 001.979 9.86 8.652 8.652 0 00.32 9.702zm0-6.117V.14c2.482 0 4.812.474 6.988 1.424 2.188.937 4.097 2.223 5.727 3.858s2.92 3.542 3.873 5.722c.941 2.191 1.412 4.523 1.412 6.996h-3.459c0-2.004-.382-3.89-1.147-5.66s-1.806-3.314-3.123-4.632a14.708 14.708 0 00-4.624-3.12C4.202 3.966 2.32 3.585.32 3.585z"/></svg>',

			'flickr'      => '<svg width="18" height="9" viewBox="0 0 18 9" xmlns="http://www.w3.org/2000/svg"><path d="M4.08 8.179c-.563 0-1.09-.105-1.582-.316a4.082 4.082 0 01-1.292-.87 4.274 4.274 0 01-.888-1.31C.108 5.19.002 4.663.002 4.1s.105-1.09.316-1.582c.211-.492.501-.923.87-1.292.37-.37.806-.665 1.31-.888A3.973 3.973 0 014.08.023c.563 0 1.09.105 1.582.316.492.211.923.501 1.292.87.37.37.665.806.888 1.31.21.492.316 1.02.316 1.582s-.105 1.09-.316 1.582c-.211.492-.501.923-.87 1.292-.37.369-.806.665-1.31.888-.492.21-1.02.316-1.582.316zm9.844 0c-.563 0-1.09-.105-1.582-.316a4.082 4.082 0 01-1.292-.87 4.274 4.274 0 01-.888-1.31c-.21-.492-.316-1.02-.316-1.582s.105-1.09.316-1.582c.211-.492.501-.923.87-1.292.37-.37.806-.665 1.31-.888a3.973 3.973 0 011.582-.316c.562 0 1.09.105 1.582.316.492.211.923.501 1.292.87.369.37.665.806.888 1.31.21.492.316 1.02.316 1.582s-.106 1.09-.316 1.582c-.211.492-.501.923-.87 1.292-.37.369-.806.665-1.31.888-.492.21-1.02.316-1.582.316z"/></svg>',

			'foursquare'  => '<svg width="14" height="19" viewBox="0 0 14 19" xmlns="http://www.w3.org/2000/svg"><path d="M12.28.101c.727 0 1.195.196 1.406.589.211.392.246.958.106 1.696-.106.492-.252 1.195-.44 2.11-.176.914-.354 1.828-.536 2.742-.182.914-.349 1.734-.5 2.46-.142.716-.224 1.126-.247 1.231-.07.27-.193.571-.37.905-.175.334-.567.501-1.177.501H7.57c-.059 0-.097.006-.114.018l-.132.088c-.023.023-.275.31-.756.861-.48.563-1.005 1.175-1.573 1.837-.568.662-1.093 1.269-1.573 1.82l-.703.826c-.176.21-.384.316-.624.316s-.42-.023-.537-.07c-.105-.035-.225-.132-.36-.29-.135-.159-.202-.413-.202-.765V1.718c0-.316.12-.665.36-1.046.24-.38.683-.571 1.327-.571h9.598zm-.055 10.055l.09-.457.06-.283-.15.74zm.15-.74l.584-2.898-.143.72a251.52 251.52 0 01-.442 2.178zm.584-2.898c.132-.674.264-1.348.393-2.023.188-.914.334-1.617.44-2.109l-.833 4.132zm-1.241-3.816a.517.517 0 00-.097-.448.45.45 0 00-.36-.184H3.562c-.176 0-.32.064-.431.193s-.167.264-.167.404v11.918c0 .012.008.018.026.018s.032-.006.044-.018l.44-.545a55.84 55.84 0 001.01-1.204c.37-.45.724-.882 1.064-1.292.328-.41.533-.662.615-.756.164-.199.302-.325.413-.378.111-.052.29-.079.536-.079h2.566c.2 0 .346-.058.44-.175.094-.118.146-.211.158-.282a35 35 0 00.202-.993c.112-.568.185-.929.22-1.08.012-.165-.038-.317-.15-.458s-.237-.21-.377-.21H6.97c-.21 0-.38-.065-.51-.194s-.193-.3-.193-.51v-.457c0-.211.065-.38.194-.51s.298-.193.51-.193H10.732c.13 0 .252-.05.37-.15s.193-.202.228-.307l.387-2.11z"/></svg>',

			'goodreads'   => '<svg width="13" height="19" viewBox="0 0 13 19" xmlns="http://www.w3.org/2000/svg"><path d="M10.63.418h1.444v12.294c0 .869-.117 1.638-.352 2.307-.247.67-.599 1.233-1.057 1.691s-1.027.804-1.708 1.04c-.681.234-1.462.351-2.343.351-.775 0-1.467-.094-2.078-.281-.61-.2-1.127-.485-1.55-.855a3.653 3.653 0 01-.969-1.347c-.234-.528-.358-1.133-.37-1.814l-.035-.388H2.88v.423c.06.951.39 1.67.995 2.158.605.487 1.495.73 2.669.73 1.409 0 2.442-.325 3.1-.977.657-.652.986-1.664.986-3.038v-1.867c-.458.634-1.075 1.133-1.85 1.497a5.769 5.769 0 01-2.483.546c-.88 0-1.67-.173-2.369-.52s-1.283-.842-1.752-1.488a6.218 6.218 0 01-.969-1.955 8.196 8.196 0 01-.334-2.378c0-.974.135-1.855.405-2.641.27-.799.651-1.48 1.144-2.044S3.51.864 4.202.56C4.893.254 5.662.101 6.508.101c.845 0 1.629.185 2.35.555a4.72 4.72 0 011.77 1.559V.418zm.317 6.13a6.213 6.213 0 00-.361-2.238c-.252-.692-.614-1.274-1.083-1.743a3.753 3.753 0 00-1.268-.846 4.137 4.137 0 00-1.55-.281c-.81 0-1.491.152-2.043.458a3.998 3.998 0 00-1.357 1.215 5.054 5.054 0 00-.74 1.638 7.39 7.39 0 00-.228 1.796c0 .764.1 1.45.3 2.061.199.61.48 1.136.845 1.576s.804.778 1.32 1.013a4.124 4.124 0 001.726.352c.764 0 1.427-.14 1.99-.422a4.218 4.218 0 001.374-1.145c.364-.47.635-1.004.81-1.603s.265-1.21.265-1.832z"/></svg>',

			'google-plus' => '<svg width="19" height="13" viewBox="0 0 19 13" xmlns="http://www.w3.org/2000/svg"><path d="M6.14 5.323h5.45c.082.516.111 1.031.088 1.547s-.112 1.02-.264 1.512a5.708 5.708 0 01-.554 1.31c-.24.416-.536.788-.888 1.116-.34.305-.709.56-1.107.765s-.815.348-1.248.43a6.43 6.43 0 01-1.495.176c-.504 0-.99-.058-1.459-.176a5.32 5.32 0 01-1.107-.439 6.98 6.98 0 01-1.002-.65 4.671 4.671 0 01-.906-.862 6.176 6.176 0 01-.677-1.037 5.985 5.985 0 01-.545-1.846A5.935 5.935 0 01.48 5.288a7.45 7.45 0 01.211-.774c.082-.246.176-.48.281-.703A5.884 5.884 0 012.334 2.08 5.754 5.754 0 014.24.963a6.02 6.02 0 011.97-.307c.68.005 1.324.12 1.933.342.352.13.686.29 1.002.484.317.193.604.419.862.677-.094.093-.19.184-.29.272-.1.088-.19.185-.273.29L8.39 3.776a2.845 2.845 0 00-1.265-.738c-.27-.094-.548-.141-.836-.141s-.57.023-.852.07c-.317.07-.627.19-.932.36-.305.17-.568.367-.791.59-.2.199-.37.416-.51.65-.14.235-.252.475-.334.72-.117.376-.176.754-.176 1.135s.059.753.176 1.116c.129.352.305.68.528.985.222.304.492.562.808.773.2.14.405.26.616.36s.427.168.65.203c.234.047.472.067.712.061.24-.006.472-.026.694-.061.211-.035.428-.097.65-.185.224-.088.429-.19.616-.308.305-.21.557-.471.756-.782.2-.31.334-.647.404-1.01H6.14V5.322zm12.236.14v1.442h-1.969v1.97h-1.442v-1.97h-1.969V5.464h1.97v-1.97h1.441v1.97h1.969z"/></svg>',

			'google'      => '<svg width="18" height="19" viewBox="0 0 18 19" xmlns="http://www.w3.org/2000/svg"><path d="M17.65 8.499c.047.633.038 1.256-.026 1.871a12.714 12.714 0 01-.325 1.819 8.868 8.868 0 01-.896 2.02 7.555 7.555 0 01-1.388 1.704 8.218 8.218 0 01-1.731 1.213 7.965 7.965 0 01-1.959.72 9.909 9.909 0 01-2.31.255 8.672 8.672 0 01-2.293-.325 9.662 9.662 0 01-1.73-.66 7.935 7.935 0 01-1.538-.992c-.515-.41-.981-.867-1.397-1.37s-.77-1.043-1.063-1.617C.772 12.692.59 12.23.45 11.75s-.246-.966-.317-1.458a7.93 7.93 0 01-.07-1.493 9.1 9.1 0 01.158-1.494c.094-.386.202-.77.325-1.15.123-.381.272-.747.448-1.099.27-.515.58-1.007.931-1.475a9.392 9.392 0 012.618-2.267 8.746 8.746 0 011.581-.72A9.498 9.498 0 019.146.1a8.34 8.34 0 011.529.14 8.067 8.067 0 011.493.422c.527.2 1.034.448 1.52.747.486.298.94.647 1.361 1.045-.14.14-.284.287-.43.44-.146.152-.296.298-.448.439-.27.28-.542.556-.817.825s-.553.545-.835.826c-.269-.27-.568-.5-.896-.694a5.15 5.15 0 00-1.036-.465c-.422-.13-.858-.2-1.31-.211s-.892.023-1.326.105a5.833 5.833 0 00-1.458.58c-.457.258-.86.568-1.212.931-.305.305-.568.636-.79.993s-.405.729-.545 1.115a5.67 5.67 0 00-.255 1.757c.006.61.102 1.195.29 1.757h-.035a.603.603 0 01.061.123c.018.047.032.088.044.123.2.492.469.958.808 1.397.34.44.726.805 1.16 1.098a6.2 6.2 0 00.957.545 4.63 4.63 0 001.046.334 5.451 5.451 0 002.143 0c.352-.059.694-.156 1.028-.29s.647-.296.94-.484c.48-.316.878-.717 1.195-1.203.316-.486.527-1.01.632-1.572H9.041V7.41h8.468c.035.175.065.357.088.544s.041.37.053.545z"/></svg>',

			'github'      => '<svg width="17" height="19" viewBox="0 0 17 19" xmlns="http://www.w3.org/2000/svg"><path d="M16.82 6.336c0 1.127-.153 2.049-.458 2.765-.317.716-.725 1.286-1.224 1.709s-1.066.728-1.7.915c-.645.188-1.291.317-1.937.388.211.2.393.478.546.837.153.358.229.807.229 1.347V18.1H6.464a173.584 173.584 0 00-.035-2.783c-1.62.353-2.683.212-3.188-.422l-.758-.951c-.27-.67-.554-1.125-.854-1.365l-.449-.361c-.528-.353-.643-.558-.343-.617l.449-.088c.587.059 1.033.273 1.338.643l.458.555c.517.892 1.127 1.318 1.832 1.277.704-.041 1.221-.138 1.55-.291.059-.376.158-.693.3-.951s.299-.458.475-.599a12.085 12.085 0 01-1.938-.405c-.634-.2-1.206-.51-1.717-.933S2.665 9.817 2.36 9.1c-.317-.716-.475-1.638-.475-2.765 0-.634.108-1.215.325-1.744.218-.528.508-.992.872-1.391-.059-.14-.132-.49-.22-1.048S2.894.911 3.223.101l.757.035C4.485.16 5.301.547 6.43 1.3A11.364 11.364 0 019.352.91c.517 0 1.022.036 1.515.106.493.07.974.164 1.444.282C13.427.536 14.24.145 14.751.128L15.515.1c.318.81.432 1.494.344 2.052s-.167.907-.238 1.048c.388.41.684.878.89 1.4.205.523.308 1.101.308 1.735zM2.096 12.078a.235.235 0 00-.026-.106.134.134 0 00-.08-.07c-.058-.036-.105-.047-.14-.036a.106.106 0 00-.071.07c-.012.024-.006.057.018.098a.163.163 0 00.088.079c.035.012.076.012.123 0s.076-.024.088-.035zm.458.528a.122.122 0 00.035-.123.261.261 0 00-.07-.124c-.036-.058-.077-.088-.124-.088s-.088.006-.123.018c-.012.035-.017.076-.017.123s.017.088.052.124.083.058.141.07c.059.012.094.012.106 0zm.458.67a.18.18 0 00.053-.133.245.245 0 00-.053-.15c-.035-.058-.076-.093-.123-.105a.22.22 0 00-.124 0c-.035.023-.053.064-.053.123s.018.112.053.159c.035.058.077.094.124.105a.22.22 0 00.123 0zm.634.633c.035-.035.044-.082.026-.14a.267.267 0 00-.097-.141c-.035-.06-.082-.091-.14-.097s-.106.014-.141.061c-.036.036-.044.08-.027.132.018.053.05.103.097.15.035.059.082.088.141.088s.106-.017.14-.053zm.88.388c.012-.059-.002-.112-.044-.159s-.085-.076-.132-.088a.469.469 0 00-.202.01c-.065.017-.103.049-.115.096a.177.177 0 00.044.159.274.274 0 00.168.088c.058.011.117.008.176-.01s.094-.05.106-.096zm.705.246c.07 0 .13-.017.176-.052s.07-.077.07-.124a.146.146 0 00-.07-.132.327.327 0 00-.176-.044c-.07 0-.13.018-.176.053-.047.035-.07.076-.07.123 0 .06.023.103.07.132.047.03.106.044.176.044zm.916-.105c.07-.024.129-.056.176-.097s.07-.08.07-.115a.223.223 0 00-.105-.123.284.284 0 00-.176-.017c-.07.023-.13.055-.176.096s-.07.08-.07.115c.023.059.058.1.105.123s.105.03.176.018z"/></svg>',

			'instagram'   => '<svg width="19" height="19" viewBox="0 0 19 19" xmlns="http://www.w3.org/2000/svg"><path d="M9.51 1.757c-1.219 0-2.01.009-2.373.026-.363.018-.791.033-1.283.044a6.324 6.324 0 00-1.046.123 4.488 4.488 0 00-.642.194c-.199.07-.38.16-.545.272a3.267 3.267 0 00-.852.853 2.815 2.815 0 00-.29.562 5.826 5.826 0 00-.176.624c-.059.252-.094.607-.105 1.064-.024.48-.041.902-.053 1.265s-.018 1.149-.018 2.356c0 1.219.009 2.01.027 2.373.017.363.032.79.044 1.283.023.445.064.794.123 1.046s.123.466.193.642c.07.199.161.38.272.544a3.267 3.267 0 00.853.853c.164.111.352.208.562.29.165.059.373.117.624.176.252.058.607.094 1.064.105.48.024.902.041 1.266.053s1.148.018 2.355.018c1.219 0 2.01-.01 2.373-.027s.791-.032 1.283-.044a6.324 6.324 0 001.046-.123c.252-.058.466-.123.642-.193.199-.07.38-.161.545-.273a3.267 3.267 0 00.852-.852c.112-.164.208-.352.29-.563.059-.164.117-.372.176-.624s.094-.606.106-1.063c.023-.48.04-.902.052-1.266s.018-1.148.018-2.355c0-1.219-.009-2.01-.027-2.373a53.213 53.213 0 01-.043-1.283 6.324 6.324 0 00-.124-1.046 4.488 4.488 0 00-.193-.642c-.07-.2-.161-.38-.272-.545s-.244-.322-.396-.474a2.83 2.83 0 00-1.02-.668 5.826 5.826 0 00-.623-.176c-.252-.059-.607-.094-1.064-.106-.48-.023-.902-.04-1.266-.052s-1.148-.018-2.355-.018zm0-1.617c1.23 0 2.036.009 2.417.026.38.018.806.032 1.274.044.493.024.909.067 1.248.132.34.064.65.15.932.255.305.129.589.275.853.44.263.163.518.368.764.615s.454.503.624.773.314.55.43.844c.106.27.194.577.265.923.07.345.11.764.123 1.256.023.48.04.909.052 1.284.012.375.018 1.177.018 2.408s-.009 2.036-.026 2.417c-.018.38-.033.805-.044 1.274a9.181 9.181 0 01-.132 1.248c-.065.34-.15.65-.255.932-.129.304-.275.589-.44.852s-.369.519-.615.765-.504.454-.773.624c-.27.17-.551.313-.844.43-.27.106-.577.194-.923.264-.346.07-.764.112-1.257.123-.48.024-.908.041-1.283.053-.375.012-1.178.018-2.408.018s-2.036-.01-2.417-.027-.806-.032-1.274-.044a9.181 9.181 0 01-1.248-.131c-.34-.065-.65-.15-.932-.255a6.136 6.136 0 01-.853-.44c-.263-.164-.518-.369-.764-.615s-.454-.504-.624-.773c-.17-.27-.314-.551-.43-.844a5.607 5.607 0 01-.265-.923A7.287 7.287 0 01.58 12.83c-.023-.48-.04-.908-.052-1.283C.516 11.173.51 10.37.51 9.14s.009-2.036.026-2.417c.018-.381.033-.806.044-1.275.024-.492.068-.908.132-1.248.065-.34.15-.65.255-.931.129-.305.275-.59.44-.853.164-.264.369-.518.615-.764s.504-.455.773-.625c.27-.17.551-.313.844-.43.27-.106.577-.194.923-.264.345-.07.764-.111 1.257-.123.48-.023.908-.04 1.283-.053S8.279.14 9.51.14zm0 4.36c.645 0 1.248.122 1.81.368a4.53 4.53 0 012.461 2.461c.247.563.37 1.166.37 1.81 0 .645-.123 1.249-.37 1.811a4.53 4.53 0 01-2.46 2.461 4.467 4.467 0 01-1.811.37 4.467 4.467 0 01-1.81-.37 4.53 4.53 0 01-2.461-2.46 4.467 4.467 0 01-.37-1.811c0-.645.123-1.248.37-1.81a4.53 4.53 0 012.46-2.462A4.467 4.467 0 019.51 4.5zm0 7.663c.832 0 1.544-.299 2.136-.896.591-.598.887-1.307.887-2.127 0-.832-.298-1.544-.896-2.136-.598-.592-1.307-.888-2.127-.888-.832 0-1.544.3-2.136.897-.592.598-.887 1.307-.887 2.127 0 .832.298 1.544.896 2.136.598.591 1.307.887 2.127.887zm5.871-7.84c0 .293-.1.548-.299.765a.982.982 0 01-.756.325 1.06 1.06 0 01-.764-.316 1.038 1.038 0 01-.325-.774c0-.292.105-.541.316-.747.21-.205.469-.307.773-.307.293 0 .542.1.747.299.206.199.308.45.308.755z"/></svg>',

			'lastfm'      => '<svg width="19" height="11" viewBox="0 0 19 11" xmlns="http://www.w3.org/2000/svg"><path d="M14.52 10.093c-.89 0-1.647-.094-2.268-.281-.609-.188-1.125-.446-1.547-.774a4.092 4.092 0 01-1.019-1.142 7.805 7.805 0 01-.615-1.389l-.668-2.074c-.246-.75-.563-1.383-.95-1.899-.386-.515-1.03-.773-1.933-.773-.633 0-1.213.258-1.74.773-.528.516-.791 1.407-.791 2.672 0 .985.222 1.764.668 2.338s1.03.862 1.758.862c.808 0 1.456-.2 1.942-.598l.73-.598.667 1.793-.852.545c-.569.363-1.433.545-2.593.545-.715 0-1.353-.105-1.916-.316-.562-.211-1.034-.519-1.415-.923s-.67-.905-.87-1.503c-.2-.598-.299-1.29-.299-2.074 0-.82.1-1.547.299-2.18.2-.645.495-1.186.888-1.626s.876-.776 1.45-1.01C4.02.225 4.688.108 5.45.108c.68 0 1.283.064 1.81.193.528.129.99.34 1.389.633.398.293.738.668 1.02 1.125.292.469.55 1.037.773 1.705l.668 2.074c.246.75.62 1.371 1.125 1.864.504.492 1.277.738 2.32.738.703 0 1.236-.085 1.6-.255.363-.17.545-.448.545-.835 0-.293-.103-.545-.308-.756-.205-.21-.571-.38-1.099-.51l-1.406-.351c-.844-.188-1.453-.504-1.828-.95-.375-.445-.562-1.019-.562-1.722 0-1.125.38-1.899 1.142-2.32s1.612-.633 2.549-.633c1.055 0 1.887.199 2.496.597s.961 1.067 1.055 2.004l-2.074.246c-.036-.433-.19-.747-.466-.94-.276-.193-.66-.29-1.152-.29-.457 0-.811.094-1.063.281-.252.188-.378.457-.378.809 0 .316.08.568.237.756.158.187.466.334.923.44l1.336.28c.879.211 1.55.548 2.013 1.011.463.463.694 1.11.694 1.942 0 1.032-.392 1.764-1.178 2.198-.785.433-1.822.65-3.11.65z"/></svg>',

			'linkedin'    => '<svg width="19" height="19" viewBox="0 0 19 19" xmlns="http://www.w3.org/2000/svg"><path d="M14.27 9.628a1.057 1.057 0 00-.492-.316 2.467 2.467 0 00-.774-.106c-.386 0-.712.059-.975.176a1.64 1.64 0 00-.642.492c-.152.223-.266.492-.343.809s-.114.68-.114 1.09v6.152a.473.473 0 01-.026.15.448.448 0 01-.114.166c-.036.035-.083.068-.141.097s-.117.044-.176.044H7.24c-.047 0-.097-.015-.15-.044s-.108-.062-.167-.097a1.305 1.305 0 01-.096-.15.339.339 0 01-.044-.166V6.57a.31.31 0 01.044-.15 5.8 5.8 0 01.096-.167c.047-.035.097-.067.15-.096s.108-.044.167-.044h3.129a.69.69 0 01.149.017c.053.012.097.03.132.053.035.012.061.041.079.088l.061.158c0 .047.006.094.018.14a.48.48 0 01.018.106v.281c.398-.386.867-.67 1.406-.852a5.59 5.59 0 011.793-.273c.75 0 1.418.088 2.004.264.586.176 1.09.451 1.511.826.422.387.739.856.95 1.407.21.55.316 1.195.316 1.933v7.664a.473.473 0 01-.026.15.448.448 0 01-.115.167c-.035.035-.082.067-.14.096s-.117.044-.176.044h-3.305a.31.31 0 01-.15-.044 5.8 5.8 0 01-.166-.096 1.305 1.305 0 01-.097-.15.339.339 0 01-.044-.167V11c0-.293-.026-.553-.079-.782a1.85 1.85 0 00-.237-.589zm-9.738-5.45c-.211.212-.446.37-.703.475s-.54.159-.844.159c-.293 0-.571-.053-.835-.159s-.501-.263-.712-.474c-.211-.211-.37-.443-.475-.695a2.19 2.19 0 01-.158-.852c0-.293.053-.566.158-.817.106-.252.264-.484.475-.695.21-.21.445-.375.703-.492s.54-.176.844-.176c.293 0 .571.059.835.176s.5.281.712.492c.21.211.369.44.474.686.106.246.159.521.159.826 0 .316-.053.6-.159.852a2.159 2.159 0 01-.474.695zm.527 2.391v11.355a.31.31 0 01-.044.15c-.03.053-.061.108-.097.167a1.305 1.305 0 01-.149.096.339.339 0 01-.167.044H1.368a.31.31 0 01-.15-.044 5.802 5.802 0 01-.167-.096 1.305 1.305 0 01-.096-.15.339.339 0 01-.044-.167V6.57a.31.31 0 01.044-.149c.029-.053.061-.108.096-.167.047-.035.097-.067.15-.097a.339.339 0 01.167-.044h3.234a.31.31 0 01.15.044 5.8 5.8 0 01.166.097c.036.047.068.097.097.15.03.052.044.108.044.166z"/></svg>',

			'mail'        => '<svg width="18" height="13" viewBox="0 0 18 13" xmlns="http://www.w3.org/2000/svg"><path d="M9.6 8.175l8.402-6.856v10.793h-18V1.32l8.403 6.856c.093.07.19.129.29.176s.202.07.307.07.211-.023.317-.07c.105-.047.199-.106.28-.176zm-.598-2.074a.662.662 0 01-.263-.053 1.417 1.417 0 01-.229-.123L1.373.089h15.258L9.495 5.925a.813.813 0 01-.492.176z"/></svg>',

			'medium'      => '<svg width="19" height="16" viewBox="0 0 19 16" xmlns="http://www.w3.org/2000/svg"><path d="M18.39 15.093h-6.856l1.372-1.512c.023-.023.038-.079.044-.167.005-.088.008-.202.008-.342V3.667l-4.5 11.426L3.572 4.441a5.822 5.822 0 01-.167-.457 39.698 39.698 0 01-.15-.474v7.927c0 .164.006.282.018.352.012.07.076.182.193.334l2.074 2.97H.39l2.092-2.97c.117-.152.178-.264.184-.334.006-.07.01-.188.01-.352V3.246c0-.176.002-.293.008-.352.006-.058-.032-.14-.114-.246L.97.222h4.852l4.2 9.3 3.657-9.3h4.711l-1.266 1.565c-.082.094-.131.187-.149.281s-.026.211-.026.352v10.635c0 .14.003.252.008.334.006.082.027.14.062.176l1.371 1.529z"/></svg>',

			'meetup'      => '<svg width="19" height="15" viewBox="0 0 19 15" xmlns="http://www.w3.org/2000/svg"><path d="M12.22 11.542c-.012-.258.012-.504.07-.738s.14-.463.246-.685c.375-.797.75-1.59 1.125-2.38a774.383 774.383 0 001.291-2.742c.053-.123.085-.254.097-.395a1.428 1.428 0 00-.018-.387c-.023-.128-.105-.228-.246-.298-.14-.07-.278-.091-.413-.062s-.255.097-.36.202c-.14.106-.255.232-.343.378a6.41 6.41 0 00-.237.43c-.27.528-.533 1.049-.79 1.564a192.161 192.161 0 01-1.582 3.092l-.773 1.547c-.082.152-.164.301-.246.448a1.8 1.8 0 01-.298.395c-.223.234-.501.39-.835.465-.334.077-.653.05-.957-.079a.87.87 0 01-.519-.535c-.088-.252-.073-.548.044-.888.2-.527.401-1.048.606-1.563s.413-1.031.624-1.547l.316-.843.317-.843c.035-.07.064-.147.087-.229s.041-.158.053-.228a1.094 1.094 0 00-.088-.703.8.8 0 00-.474-.421.77.77 0 00-.598.07c-.199.105-.35.28-.456.527-.246.609-.492 1.221-.738 1.836s-.486 1.227-.72 1.836c-.258.632-.513 1.268-.765 1.906s-.506 1.268-.764 1.889c-.281.68-.68 1.13-1.195 1.353s-1.118.222-1.81 0c-.609-.2-1.068-.563-1.379-1.09-.31-.527-.377-1.112-.202-1.756.13-.528.27-1.055.422-1.582.152-.527.31-1.051.474-1.572.164-.522.328-1.04.492-1.555.176-.527.346-1.048.51-1.564l.457-1.265c.153-.422.316-.837.491-1.247.305-.656.794-1.166 1.467-1.529.674-.363 1.327-.457 1.96-.28.28.07.553.172.816.307s.525.272.782.412c.27.153.522.205.756.159.234-.047.462-.164.685-.352a92.698 92.698 0 00.597-.527c.422-.34.864-.504 1.327-.492.463.012.893.2 1.291.562.07.059.138.118.202.176.065.059.132.117.202.176a.653.653 0 00.378.202.597.597 0 00.413-.114 2.279 2.279 0 011.15-.37c.393-.011.788.024 1.187.106.55.117.978.413 1.282.887s.416.993.334 1.555a7.778 7.778 0 01-.588 2.02 277.748 277.748 0 00-1.626 3.725 9.284 9.284 0 00-.58 1.863c-.011.117-.023.228-.035.334a3.025 3.025 0 00-.017.333c-.012.328.05.583.184.765.135.181.372.313.712.395.21.059.43.105.659.14s.454.077.676.124c.316.07.539.184.668.342.129.158.187.372.175.641-.011.246-.09.446-.237.598s-.372.258-.676.316c-.55.106-1.092.123-1.625.053s-1.052-.217-1.555-.44c-.598-.257-1.051-.629-1.362-1.115-.31-.486-.483-1.057-.518-1.713z"/></svg>',

			'pinterest'   => '<svg width="15" height="19" viewBox="0 0 15 19" xmlns="http://www.w3.org/2000/svg"><path d="M4.77 7.548c0-.387.047-.738.14-1.055.094-.316.247-.61.458-.879.187-.257.404-.457.65-.597.246-.14.51-.211.791-.211.223 0 .422.035.598.105.175.07.328.188.457.352a1.866 1.866 0 01.387 1.16c0 .14-.015.296-.044.466-.03.17-.074.354-.132.554-.047.21-.1.436-.159.676s-.134.495-.228.765c-.082.281-.158.527-.229.738-.07.211-.123.399-.158.563-.07.28-.079.545-.026.79.053.247.167.475.343.686.175.188.378.334.606.44s.483.158.765.158c.492 0 .94-.138 1.344-.413s.765-.689 1.082-1.24c.316-.562.553-1.177.711-1.845s.238-1.4.238-2.197c0-.598-.097-1.143-.29-1.635a3.831 3.831 0 00-.87-1.319 3.932 3.932 0 00-1.398-.835c-.545-.193-1.169-.29-1.872-.29-.785 0-1.494.123-2.127.37a4.48 4.48 0 00-1.67 1.142A4.99 4.99 0 003.03 5.632a5.075 5.075 0 00-.37 1.951c0 .422.06.809.176 1.16.118.352.293.668.528.95a.566.566 0 01.158.28.96.96 0 01.017.317c-.035.082-.07.194-.105.334s-.07.31-.105.51a.397.397 0 01-.07.176.32.32 0 01-.141.105.614.614 0 01-.15.018.852.852 0 01-.167-.018 3.17 3.17 0 01-.949-.606 3.555 3.555 0 01-.703-.94A4.85 4.85 0 01.727 8.69a6.23 6.23 0 01-.14-1.354c0-.316.026-.627.078-.932.053-.304.132-.615.238-.931.105-.317.231-.624.378-.923s.325-.589.536-.87c.21-.293.445-.566.703-.818.258-.252.527-.495.809-.73.292-.222.618-.424.975-.606a7.928 7.928 0 011.204-.483c.422-.117.859-.21 1.31-.281.451-.07.917-.106 1.397-.106.504 0 .988.044 1.45.132.463.088.906.22 1.328.396.421.152.814.342 1.177.571.364.229.686.483.967.765.293.293.55.6.774.923.222.322.421.67.597 1.045.153.364.273.739.36 1.125s.132.774.132 1.16c0 1.055-.131 2.02-.395 2.892a6.954 6.954 0 01-1.222 2.347c-.539.703-1.154 1.225-1.846 1.564-.69.34-1.459.51-2.302.51-.281 0-.554-.03-.818-.088a2.728 2.728 0 01-.764-.299 2.626 2.626 0 01-.633-.413 1.94 1.94 0 01-.422-.536c-.187.82-.343 1.477-.466 1.969s-.214.82-.272.984a7.114 7.114 0 01-.528 1.336 13.037 13.037 0 01-.984 1.618H3.61a15.553 15.553 0 01-.08-1.97c.018-.597.08-1.136.185-1.616L5.086 9.27a3.083 3.083 0 01-.237-.765 5.323 5.323 0 01-.08-.958z"/></svg>',

			'pocket'      => '<svg width="19" height="17" viewBox="0 0 19 17" xmlns="http://www.w3.org/2000/svg"><path d="M18.2 2.721v4.673c0 .492-.03.99-.088 1.493-.058.504-.158.99-.299 1.459-.269.831-.647 1.607-1.133 2.327s-1.068 1.35-1.748 1.89a8.416 8.416 0 01-1.142.807c-.398.246-.814.457-1.247.633s-.878.316-1.335.421a8.79 8.79 0 01-1.37.211c-.48.035-.955.03-1.424-.017a7.998 7.998 0 01-1.405-.229 9.677 9.677 0 01-1.353-.457 8.131 8.131 0 01-1.265-.667 9.857 9.857 0 01-1.985-1.704A8.739 8.739 0 01.983 11.4a9.283 9.283 0 01-.492-1.388 7.71 7.71 0 01-.246-1.423C.21 8.062.195 7.535.201 7.008s.01-1.043.01-1.546V2.3c0-.387.107-.738.324-1.055a1.55 1.55 0 01.87-.632c.105-.035.213-.056.325-.062a6.52 6.52 0 01.342-.008h8.714c.539.011 1.072.017 1.599.017.527 0 1.06.006 1.599.018h1.58c.235-.024.478-.035.73-.035s.5.035.746.105a1.956 1.956 0 01.914.703c.14.21.217.43.228.659.012.228.018.465.018.711zm-3.584 4.076c.07-.2.082-.404.035-.615s-.129-.392-.246-.545a1.232 1.232 0 00-.527-.36 1.236 1.236 0 00-.632-.061 1.4 1.4 0 00-.72.386c-.2.2-.393.387-.58.562a142.695 142.695 0 01-2.74 2.6 21.07 21.07 0 01-1.187-1.097c-.38-.381-.77-.765-1.168-1.151l-.29-.29c-.1-.1-.202-.19-.307-.272a4.673 4.673 0 00-.598-.563c-.164-.105-.351-.166-.562-.184s-.41.009-.597.08c-.422.175-.66.485-.712.93-.052.445.068.814.36 1.107.41.41.817.808 1.221 1.195.404.386.806.773 1.204 1.16.199.175.395.36.588.552.194.194.396.384.606.571.164.164.337.34.519.527.181.188.395.305.64.352a1.308 1.308 0 001.16-.316 62.497 62.497 0 001.23-1.195 111.392 111.392 0 002.425-2.284c.164-.164.34-.33.527-.5.187-.17.304-.367.351-.59z"/></svg>',

			'reddit'      => '<svg width="19" height="16" viewBox="0 0 19 16" xmlns="http://www.w3.org/2000/svg"><path d="M18.19 7.429c0 .375-.105.709-.316 1.002-.211.293-.48.51-.809.65a2.953 2.953 0 01.105.774c0 .726-.21 1.406-.632 2.039-.422.633-.994 1.186-1.714 1.661-.721.475-1.568.853-2.54 1.134a11.8 11.8 0 01-3.13.404c-1.112 0-2.155-.14-3.128-.422-.973-.27-1.82-.641-2.54-1.116-.72-.475-1.292-1.028-1.714-1.661-.422-.645-.633-1.324-.633-2.04 0-.14.009-.28.027-.421.017-.14.044-.27.079-.387a1.753 1.753 0 01-.765-.65 1.719 1.719 0 01-.29-.967c0-.492.17-.908.51-1.248.34-.34.756-.51 1.248-.51.281 0 .536.059.764.176.229.117.42.27.572.457a8.346 8.346 0 011.142-.668c.41-.2.844-.369 1.301-.51.457-.14.938-.252 1.442-.334a11.25 11.25 0 011.564-.14L10.245.257a.295.295 0 01.15-.167.377.377 0 01.236-.044l3.551.844c.024 0 .041.006.053.018l.018.017c.128-.246.304-.442.527-.589s.48-.22.773-.22c.387 0 .712.132.976.396s.395.589.395.976c0 .386-.131.718-.395.993a1.297 1.297 0 01-.976.413c-.386 0-.714-.132-.984-.396a1.335 1.335 0 01-.422-.94h-.14L10.771.785l-1.3 3.867a11.692 11.692 0 013.103.475c.473.14.916.316 1.326.527.422.199.809.427 1.16.685a1.705 1.705 0 011.371-.668c.492 0 .908.17 1.248.51.34.34.51.756.51 1.248zM5.077 8.835c0 .352.129.656.387.914s.562.387.914.387c.351 0 .656-.129.914-.387s.387-.562.387-.914c0-.351-.13-.656-.387-.914s-.563-.387-.914-.387c-.352 0-.656.13-.914.387s-.387.563-.387.914zm7.031 3.586c.035-.035.053-.085.053-.15a.27.27 0 00-.053-.166c-.035-.035-.082-.053-.14-.053s-.106.018-.141.053c-.27.27-.624.466-1.064.589-.439.123-.963.184-1.573.184-.633 0-1.166-.064-1.6-.193s-.779-.322-1.036-.58c-.036-.035-.082-.053-.141-.053s-.106.018-.14.053a.34.34 0 00-.053.175.19.19 0 00.052.141c.305.305.698.527 1.178.668s1.06.211 1.74.211c.692 0 1.278-.073 1.758-.22s.867-.366 1.16-.659zm-.07-2.285c.351 0 .65-.126.896-.378s.37-.56.37-.923c0-.351-.124-.656-.37-.914s-.545-.387-.896-.387c-.375 0-.692.13-.95.387s-.386.563-.386.914c0 .375.132.686.395.932.264.246.578.369.94.369z"/></svg>',

			'skype'       => '<svg width="19" height="19" viewBox="0 0 19 19" xmlns="http://www.w3.org/2000/svg"><path d="M17.6 11.315a4.903 4.903 0 01.492 2.18c0 .68-.129 1.318-.387 1.916a5.054 5.054 0 01-1.045 1.564c-.44.446-.958.797-1.556 1.055a4.661 4.661 0 01-1.899.387c-.421 0-.832-.053-1.23-.159a4.512 4.512 0 01-1.09-.439 5.48 5.48 0 01-.782.114c-.252.018-.507.027-.765.027a8.246 8.246 0 01-3.304-.668 8.475 8.475 0 01-2.681-1.793 8.486 8.486 0 01-1.82-2.69 8.121 8.121 0 01-.667-3.287c0-.305.017-.598.052-.879.036-.281.077-.562.123-.844C.831 7.448.666 7.06.55 6.63S.373 5.772.373 5.338c0-.68.13-1.318.387-1.916a5.054 5.054 0 011.046-1.564c.44-.445.958-.797 1.556-1.055A4.66 4.66 0 015.26.417c.492 0 .958.067 1.398.202.44.134.852.313 1.239.536a9.444 9.444 0 011.441-.106c1.172 0 2.268.223 3.287.668a8.486 8.486 0 012.69 1.82 8.475 8.475 0 011.793 2.68 8.246 8.246 0 01.668 3.305c0 .305-.012.61-.035.914-.024.305-.07.598-.14.879zm-4.008 2.215a3.214 3.214 0 00.598-1.899c0-.28-.03-.55-.088-.808a2.15 2.15 0 00-.264-.668 2.261 2.261 0 00-.395-.563 3.334 3.334 0 00-.554-.457 4.657 4.657 0 00-.66-.351c-.24-.106-.5-.211-.781-.317a9.03 9.03 0 00-.897-.246c-.316-.07-.638-.14-.967-.21l-.72-.185c-.2-.053-.358-.097-.475-.132a2.172 2.172 0 01-.334-.106 9.622 9.622 0 01-.334-.14 2.74 2.74 0 01-.29-.185.825.825 0 01-.202-.202.523.523 0 01-.14-.237.988.988 0 01-.036-.255c0-.164.044-.314.132-.448s.22-.255.395-.36c.176-.13.387-.22.633-.273a4.05 4.05 0 01.844-.08c.316 0 .595.027.835.08.24.053.43.132.571.237.14.106.273.237.396.396.123.158.237.342.343.553.105.14.199.27.28.387.083.117.165.211.247.281.094.07.2.123.316.158.118.036.246.053.387.053.176 0 .34-.03.492-.088a.99.99 0 00.387-.263c.129-.13.22-.261.272-.396.053-.135.08-.272.08-.413a2.13 2.13 0 00-.247-.984 2.612 2.612 0 00-.351-.475 3.635 3.635 0 00-.492-.44 3.688 3.688 0 00-.624-.395 4.436 4.436 0 00-.783-.308 4.034 4.034 0 00-.896-.193 10.103 10.103 0 00-1.072-.053c-.48 0-.923.03-1.328.088a4.956 4.956 0 00-1.133.3 3.913 3.913 0 00-.967.474c-.27.187-.498.404-.686.65-.199.246-.348.516-.448.809-.1.293-.15.597-.15.914 0 .34.05.65.15.931s.237.534.413.756c.176.235.387.44.633.615s.516.323.809.44c.304.14.647.267 1.028.378.38.111.788.208 1.222.29.316.07.606.14.87.21.263.071.489.13.676.177.176.07.34.143.493.22.152.076.28.166.386.272.13.105.22.228.273.369.053.14.079.287.079.44 0 .21-.053.404-.158.58s-.264.333-.475.474c-.21.164-.463.281-.756.352-.293.07-.615.105-.967.105-.246 0-.474-.017-.685-.053a2.144 2.144 0 01-.545-.158 2.167 2.167 0 01-.958-.949 16.982 16.982 0 00-.237-.457 1.965 1.965 0 00-.211-.43 1.295 1.295 0 00-.282-.308.876.876 0 00-.342-.194 1.434 1.434 0 00-.396-.052c-.176 0-.337.026-.483.079s-.279.132-.396.237a.869.869 0 00-.272.37c-.053.14-.08.286-.08.439 0 .246.045.495.133.747.087.252.22.506.395.764.176.27.384.507.624.712.24.205.501.378.782.519.41.21.867.369 1.371.474s1.055.159 1.653.159c.515 0 .993-.039 1.432-.115.44-.076.841-.19 1.204-.342a4.995 4.995 0 001.011-.563c.287-.21.525-.457.712-.738z"/></svg>',

			'slideshare'  => '<svg width="18" height="19" viewBox="0 0 18 19" xmlns="http://www.w3.org/2000/svg"><path d="M6.5 5.901c.586 0 1.084.193 1.494.58.41.387.615.85.615 1.389 0 .55-.208 1.016-.624 1.397s-.91.572-1.485.572a2.098 2.098 0 01-1.494-.58c-.41-.387-.615-.85-.615-1.39 0-.55.208-1.016.624-1.397.416-.38.91-.571 1.485-.571zm4.992-.035c.598 0 1.105.193 1.52.58.417.387.625.85.625 1.389 0 .562-.208 1.037-.624 1.424-.416.386-.923.58-1.52.58a2.098 2.098 0 01-1.495-.58 1.88 1.88 0 01-.615-1.424c0-.551.208-1.017.624-1.398s.911-.571 1.485-.571zm5.555 3.129c.27-.188.475-.19.615-.009s.123.396-.052.642c-.317.386-.78.808-1.39 1.265-.608.457-1.376.88-2.302 1.266.492 1.676.516 3.006.07 3.99-.445.997-1.028 1.662-1.749 1.996-.72.334-1.42.36-2.1.079-.668-.293-.99-.873-.967-1.74v-3.27a2.404 2.404 0 00-.492-.106v3.375c.023.868-.3 1.448-.967 1.74-.68.282-1.38.256-2.1-.078-.721-.334-1.299-1-1.732-1.996-.446-.984-.416-2.314.088-3.99-.95-.387-1.732-.809-2.347-1.266-.615-.457-1.087-.879-1.415-1.265-.153-.246-.167-.46-.044-.642.123-.182.325-.179.606.009l.115.079c.04.03.061.05.061.062V2.174c0-.492.173-.908.519-1.248.345-.34.735-.51 1.169-.51h12.62c.446 0 .81.17 1.09.51s.422.756.422 1.248v6.962a.423.423 0 00.185-.08c.03-.029.062-.05.097-.061zm-1.16.668V2.948c0-.563-.085-.973-.255-1.23-.17-.258-.519-.387-1.046-.387H3.406c-.562 0-.937.117-1.125.351S2 2.338 2 2.948v6.75c.55.316 1.096.55 1.635.703.539.14 1.051.238 1.538.29.486.053.928.074 1.327.062.41-.012.75-.024 1.02-.035.246-.012.439.005.58.052.14.047.263.112.369.194.023.023.038.044.044.061s.014.032.026.044c.106.094.211.182.317.264.105.082.199.158.28.228a1.01 1.01 0 01.282-.65c.164-.164.457-.229.879-.193.281.011.621.023 1.02.035.398.012.837-.009 1.318-.062a9.87 9.87 0 001.547-.307 7.72 7.72 0 001.705-.721z"/></svg>',

			'snapchat'    => '<svg width="18" height="18" viewBox="0 0 18 18" xmlns="http://www.w3.org/2000/svg"><path d="M17.6 13.58c.105.012.196.056.272.131s.102.173.079.29a.639.639 0 01-.036.21c-.035.105-.102.196-.201.272s-.23.16-.394.254c-.2.082-.444.16-.737.236s-.63.15-1.016.22c-.024.035-.044.09-.062.166s-.032.15-.043.22a2.855 2.855 0 00-.106.42.432.432 0 01-.175.236.507.507 0 01-.28.08H14.62a1.03 1.03 0 01-.245-.036 2.836 2.836 0 00-.42-.079 4.292 4.292 0 00-.492-.026c-.128 0-.245.009-.35.026a4.775 4.775 0 01-.35.044c-.211.035-.415.108-.614.22a7.67 7.67 0 00-.649.411 7.636 7.636 0 01-1.043.666c-.391.21-.861.316-1.41.316H8.8c-.55 0-1.02-.108-1.411-.324a11.522 11.522 0 01-1.06-.658 9.375 9.375 0 00-.64-.403 1.88 1.88 0 00-.605-.228 4.775 4.775 0 01-.35-.044 1.916 1.916 0 00-.316-.026c-.222 0-.409.015-.56.044s-.281.05-.386.061c-.082.024-.158.041-.228.053s-.135.017-.193.017c-.152 0-.26-.04-.324-.122s-.12-.158-.167-.228a2.687 2.687 0 00-.105-.42 3.64 3.64 0 00-.044-.22.526.526 0 00-.061-.166c-.386-.07-.725-.14-1.017-.21s-.537-.153-.736-.246a2.321 2.321 0 01-.386-.254.673.673 0 01-.21-.272.188.188 0 01-.035-.105V14c-.023-.094.003-.181.079-.263a.426.426 0 01.272-.14c.268-.059.531-.132.788-.22.257-.087.497-.213.719-.376.222-.129.42-.278.596-.447.175-.17.345-.348.508-.535.292-.339.514-.649.666-.929a6.92 6.92 0 00.246-.473h.035a.999.999 0 00.105-.342.574.574 0 00-.035-.29c-.07-.151-.205-.268-.403-.35-.199-.082-.38-.152-.544-.21-.058-.012-.11-.03-.157-.053s-.088-.04-.123-.052a3.966 3.966 0 01-.386-.184.996.996 0 01-.28-.22c-.152-.116-.237-.23-.254-.341a.877.877 0 01-.009-.272.74.74 0 01.14-.254 1.15 1.15 0 01.263-.237c.059-.035.137-.067.237-.096.099-.03.195-.044.289-.044.047 0 .096.006.149.018.052.011.108.029.166.052.14.07.278.123.412.158.135.035.254.053.36.053.07 0 .131-.006.184-.018a.644.644 0 00.149-.053.7.7 0 00-.027-.184.834.834 0 01-.026-.219 33.536 33.536 0 01-.052-1.928c0-.678.105-1.244.315-1.7.14-.304.3-.587.482-.85.181-.263.383-.494.605-.692.187-.2.388-.368.604-.509.217-.14.447-.268.693-.385.432-.187.835-.307 1.21-.36s.66-.079.858-.079H9.152c.187 0 .467.024.84.07s.772.17 1.193.369c.245.117.476.245.692.385.216.14.418.31.605.509.234.198.444.432.63.7.188.27.351.562.492.877.198.42.295.976.289 1.665-.006.69-.032 1.332-.079 1.928v.246a.614.614 0 01-.035.192c.047 0 .088.006.123.018a.456.456 0 00.14.018c.128 0 .254-.018.377-.053.122-.035.254-.088.394-.158.047-.035.105-.055.175-.061s.129-.009.176-.009c.07 0 .14.006.21.018a.93.93 0 01.193.052c.163.07.304.16.42.272a.526.526 0 01.176.394.721.721 0 01-.044.22.598.598 0 01-.184.254 1.444 1.444 0 01-.307.227c-.123.07-.266.135-.43.193a3.582 3.582 0 00-.122.044.601.601 0 00-.123.061c-.175.059-.365.129-.57.21-.204.082-.341.2-.411.351a.494.494 0 00-.027.28c.018.106.062.223.132.351l.228.456c.14.269.356.575.648.92.292.345.655.67 1.087.973.432.315.952.526 1.56.631z"/></svg>',

			'soundcloud'  => '<svg width="19" height="9" viewBox="0 0 19 9" xmlns="http://www.w3.org/2000/svg"><path d="M1.3 7.415H.948v-2.18H1.3v2.18zm.914.703h-.352v-3.41h.352v3.41zm.703.211h-.351V4.356h.351V8.33zm.738.176h-.351V4.532h.351v3.973zm.88 0h-.352V3.267h.351v5.238zm.738 0H4.92V2.563h.352v5.942zm.879 0H5.8V2.177h.352v6.328zm.738 0h-.352V2h.352v6.504zm.914 0h-.387V2.177h.387v6.328zm.703 0h-.352V2.388h.352v6.117zm.738 0H8.86V1.649h.386v6.856zm.88 0h-.352V1.122h.351v7.383zm6.609-.07H10.58a.303.303 0 01-.193-.07.216.216 0 01-.088-.177v-7.03c0-.071.018-.127.053-.168.035-.04.1-.079.193-.114.211-.082.437-.144.677-.185.24-.04.483-.061.73-.061.503 0 .984.094 1.44.281.458.176.862.422 1.214.738.351.317.632.692.843 1.125.223.434.358.903.405 1.407.14-.047.28-.088.422-.123.14-.036.293-.053.457-.053.609 0 1.13.214 1.564.641s.65.953.65 1.574c0 .609-.213 1.13-.641 1.564-.428.434-.952.65-1.573.65z"/></svg>',

			'spotify'     => '<svg width="19" height="13" viewBox="0 0 19 13" xmlns="http://www.w3.org/2000/svg"><path d="M14.7 12.008a.737.737 0 01-.474.342.697.697 0 01-.58-.096c-.75-.445-1.54-.808-2.37-1.089-.844-.27-1.728-.457-2.653-.562s-1.896-.117-2.915-.035a20 20 0 00-3.161.492.83.83 0 01-.588-.115.795.795 0 01-.36-.483.746.746 0 01.105-.588.83.83 0 01.492-.36 24.104 24.104 0 013.477-.527 17.117 17.117 0 013.205.053c1.025.117 2 .334 2.924.65.937.304 1.82.714 2.652 1.23.187.105.305.263.351.473s.012.416-.105.615zm1.546-3.442a.955.955 0 01-.598.43.94.94 0 01-.737-.114 13.108 13.108 0 00-2.845-1.265 16.552 16.552 0 00-3.223-.702 17.724 17.724 0 00-3.346-.07c-1.124.07-2.189.257-3.196.562-.246.082-.489.058-.729-.07s-.407-.317-.5-.563a1.03 1.03 0 01.079-.755.876.876 0 01.588-.474 16.48 16.48 0 013.618-.667c1.253-.094 2.5-.07 3.74.07a17.96 17.96 0 013.619.79c1.17.375 2.242.86 3.213 1.458.223.152.37.36.44.623.07.264.029.513-.123.747zm1.756-4.004c-.153.28-.39.46-.712.535a1.24 1.24 0 01-.904-.114c-1.019-.609-2.172-1.083-3.46-1.423-1.3-.327-2.617-.544-3.951-.65s-2.635-.105-3.9 0c-1.276.106-2.4.305-3.371.598a1.055 1.055 0 01-.878-.07 1.185 1.185 0 01-.598-.703 1.215 1.215 0 01.106-.904c.152-.287.386-.478.702-.571C2.15.92 3.42.686 4.847.557S7.74.426 9.237.55s2.975.377 4.427.763c1.451.399 2.769.943 3.951 1.634.281.175.46.418.536.729.076.31.026.606-.15.887z"/></svg>',

			'stumbleupon' => '<svg width="19" height="15" viewBox="0 0 19 15" xmlns="http://www.w3.org/2000/svg"><path d="M9.2 3.602c-.258 0-.48.09-.668.272s-.281.408-.281.677v5.766a4.118 4.118 0 01-.334 1.547c-.211.48-.501.9-.87 1.257-.37.357-.794.641-1.275.852a3.93 3.93 0 01-1.564.316 3.93 3.93 0 01-1.565-.316 4.033 4.033 0 01-1.265-.87c-.364-.37-.65-.8-.862-1.292A3.93 3.93 0 01.2 10.246V7.75h3.094v2.461c0 .258.09.48.272.668a.904.904 0 00.677.281c.258 0 .48-.09.668-.272a.904.904 0 00.281-.677v-5.8c.012-.54.13-1.044.352-1.512.21-.469.498-.88.861-1.23A4.01 4.01 0 017.671.841 3.87 3.87 0 019.2.543a4.009 4.009 0 012.795 1.125c.362.352.65.762.86 1.23.211.47.328.973.352 1.512v1.301l-1.828.528-1.23-.563V4.551c0-.258-.092-.48-.273-.668a.904.904 0 00-.677-.281zm9 6.644a3.93 3.93 0 01-.316 1.565c-.211.492-.498.923-.862 1.292-.363.37-.785.66-1.265.87-.493.211-1.014.317-1.565.317s-1.072-.106-1.564-.317c-.48-.21-.905-.498-1.275-.861a4.016 4.016 0 01-1.204-2.83V7.786l1.23.562 1.829-.562v2.53c0 .259.09.481.272.669a.904.904 0 00.677.281c.258 0 .48-.09.668-.272a.904.904 0 00.281-.677V7.75H18.2v2.496z"/></svg>',

			'tumblr'      => '<svg width="12" height="19" viewBox="0 0 12 19" xmlns="http://www.w3.org/2000/svg"><path d="M.1 8.239V5.673a5.138 5.138 0 001.828-.985c.246-.21.472-.445.677-.703s.378-.539.518-.844c.153-.293.282-.632.387-1.02.106-.386.188-.808.246-1.265h2.567v4.535h4.289V8.24h-4.29v4.606c0 .527.015.958.045 1.292.029.334.073.57.131.712a.99.99 0 00.229.395c.105.111.228.214.37.308.21.117.43.21.658.28.229.071.466.106.712.106.457 0 .914-.076 1.371-.228.457-.152.914-.381 1.371-.686v2.848c-.386.176-.755.331-1.107.466a7.137 7.137 0 01-1.002.307 7.26 7.26 0 01-.958.159c-.346.035-.694.052-1.046.052a8.03 8.03 0 01-1.169-.079 5.926 5.926 0 01-1.01-.237c-.317-.105-.613-.228-.888-.37s-.519-.31-.73-.509a3.638 3.638 0 01-.571-.598 3.28 3.28 0 01-.378-.632 3.428 3.428 0 01-.185-.791 8.727 8.727 0 01-.061-1.108V8.24H.1z"/></svg>',

			'twitch'      => '<svg width="18" height="19" viewBox="0 0 18 19" xmlns="http://www.w3.org/2000/svg"><path d="M12.8 16.061H9.284L6.93 18.417h-2.32V16.06H.284V3.545L1.48.416h16.03v10.97L12.8 16.06zm3.129-5.484V1.963H3.027v11.355h3.515v2.356l2.356-2.356h4.29l2.741-2.742zm-2.742-5.45v4.676H11.64V5.127h1.547zm-5.871 0h1.582v4.676H7.316V5.127z"/></svg>',

			'twitter'     => '<svg width="19" height="16" viewBox="0 0 19 16" xmlns="http://www.w3.org/2000/svg"><path d="M16.6 4.2v.457c0 1.219-.229 2.455-.686 3.709a10.568 10.568 0 01-2.021 3.375c-.89 1.008-1.992 1.834-3.305 2.479-1.3.632-2.8.949-4.5.949-.515 0-1.025-.035-1.53-.106-.503-.07-.992-.175-1.467-.316s-.934-.317-1.38-.527a9.099 9.099 0 01-1.283-.704 9.86 9.86 0 00.879.036c.855 0 1.679-.135 2.47-.405a6.85 6.85 0 002.135-1.177c-.808 0-1.529-.244-2.162-.73a3.782 3.782 0 01-1.318-1.837c.117.035.231.056.343.062s.231.008.36.008a4.54 4.54 0 00.984-.105A3.632 3.632 0 012.01 8.085a3.563 3.563 0 01-.844-2.338v-.07c.246.14.507.252.783.334s.565.123.87.123a3.58 3.58 0 01-1.196-1.292 3.56 3.56 0 01-.316-2.734c.094-.316.223-.615.387-.896.445.562.949 1.072 1.511 1.53a10.59 10.59 0 003.85 1.985c.727.2 1.47.317 2.232.352a3.555 3.555 0 01-.105-.844c0-.504.094-.984.281-1.441.2-.446.466-.835.8-1.17a3.625 3.625 0 011.169-.781c.445-.2.926-.3 1.441-.3.54 0 1.04.106 1.503.317.463.211.865.492 1.204.844a7.969 7.969 0 001.23-.343 6.622 6.622 0 001.126-.536c-.14.422-.352.809-.633 1.16-.281.352-.61.645-.984.88a8.26 8.26 0 001.072-.212 5.848 5.848 0 001.037-.386A8.42 8.42 0 0116.6 4.2z"/></svg>',

			'vimeo'       => '<svg width="19" height="16" viewBox="0 0 19 16" xmlns="http://www.w3.org/2000/svg"><path d="M12.1 3.883c-.21 0-.428.026-.65.08a3.818 3.818 0 00-.685.236c.222-.726.492-1.359.808-1.897s.685-.984 1.107-1.336.896-.609 1.423-.773 1.101-.234 1.722-.21c.926.01 1.608.315 2.047.913s.636 1.488.589 2.67a5.986 5.986 0 01-.281 1.442c-.164.515-.399 1.072-.703 1.669s-.685 1.242-1.142 1.933c-.445.68-.967 1.4-1.564 2.161-.61.785-1.195 1.476-1.757 2.074s-1.102 1.098-1.617 1.502c-.515.404-1.007.706-1.476.905-.469.2-.914.299-1.335.299-.528 0-1.01-.243-1.45-.73-.44-.486-.846-1.215-1.221-2.187-.246-.89-.49-1.778-.73-2.662a251.3 251.3 0 01-.711-2.68c-.281-.96-.565-1.681-.852-2.161-.287-.48-.589-.72-.905-.72-.07 0-.22.07-.448.21s-.548.351-.958.633l-.843-1.09A125.804 125.804 0 013.103 1.81C3.677 1.306 4.19.92 4.641.65 5.09.38 5.47.228 5.774.193c.68-.07 1.236.132 1.67.606.433.475.732 1.233.896 2.276.175 1.101.325 1.994.448 2.68.123.685.22 1.162.29 1.432.187.913.395 1.599.623 2.055s.466.686.712.686c.188 0 .422-.147.703-.44.281-.292.61-.732.984-1.317.363-.598.65-1.125.861-1.582s.334-.843.369-1.16c.047-.503-.032-.887-.237-1.15-.205-.264-.536-.396-.993-.396z"/></svg>',

			'vk'          => '<svg width="19" height="11" viewBox="0 0 19 11" xmlns="http://www.w3.org/2000/svg"><path d="M18.1 9.122c.105.129.234.31.387.545.152.234.222.44.21.615-.07.504-.55.691-1.44.563s-1.536-.106-1.934.07h-.07a2.677 2.677 0 01-1.187-.554 7.723 7.723 0 01-.888-.852 7.918 7.918 0 01-.58-.668c-.222-.281-.462-.399-.72-.352-.387.059-.563.337-.528.835.036.498-.035.888-.21 1.169-.13.234-.422.354-.88.36-.456.006-.878-.008-1.265-.044-1.218-.14-2.18-.498-2.882-1.072a12.286 12.286 0 01-1.864-1.88 1.12 1.12 0 01-.14-.167 3.297 3.297 0 00-.106-.15.426.426 0 01-.052-.061.085.085 0 01-.018-.044c-.035-.035-.07-.076-.105-.123s-.059-.088-.07-.123c-.024-.024-.042-.044-.053-.062s-.018-.032-.018-.044a.54.54 0 01-.088-.114 1.285 1.285 0 00-.088-.132c-.023 0-.038-.009-.044-.026a.11.11 0 00-.026-.044 5.18 5.18 0 01-.106-.185 1.384 1.384 0 00-.105-.167 4.165 4.165 0 00-.211-.316.427.427 0 01-.053-.062.085.085 0 01-.017-.044.47.47 0 01-.088-.123.47.47 0 00-.088-.123.108.108 0 00-.018-.061.085.085 0 01-.017-.044.47.47 0 01-.088-.123.47.47 0 00-.088-.123c0-.024-.006-.044-.018-.062s-.017-.032-.017-.044a27.3 27.3 0 01-1.195-2.214C1.19 2.794 1.05 2.5.91 2.196a2.32 2.32 0 01-.211-.984c.2-.152.451-.225.756-.22.304.007.615.01.931.01.282 0 .604-.01.967-.027s.65.027.861.132c.106.059.217.22.334.483s.229.49.334.677c.14.34.29.647.448.923.159.275.314.542.466.8.024.023.044.05.062.079a.34.34 0 00.044.061c0 .024.003.038.008.044l.027.027c0 .023.009.043.026.061l.044.044.018.018.017.017c.024.035.038.065.044.088s.015.041.027.053l.017.017.018.018c.023.035.044.064.061.088s.032.053.044.088c.13.164.278.36.448.588s.36.326.572.29c.21-.035.34-.251.386-.65.047-.398.07-.756.07-1.072v-.105c.012-.375.007-.745-.017-1.108-.023-.363-.088-.638-.193-.826a.665.665 0 00-.396-.343 2.02 2.02 0 01-.518-.255C6.769.908 7.04.72 7.422.65c.38-.07.823-.105 1.327-.105.422 0 .882.011 1.38.035.498.023.823.164.975.422.176.304.238.682.185 1.133-.053.451-.08.87-.08 1.257a.217.217 0 00-.034.123v.65l-.018.01c-.012.006-.017.014-.017.026v.175c0 .024.005.035.017.035h.018V4.798c.023 0 .035.003.035.009v.132c.023 0 .035.006.035.017V5.044l.009.009.026.026v.071h.035v.07c.024 0 .035.003.035.009v.026c0 .024.006.035.018.035h.018v.035l.017.01c.012.005.018.014.018.026.023 0 .035.005.035.017v.018l.035.035.018.009c.011.006.017.014.017.026H11.561v.035c.024 0 .038.003.044.009l.027.026a.855.855 0 00.562-.316c.14-.176.27-.352.387-.527.023-.035.044-.065.061-.088a.513.513 0 01.044-.053l.01-.018c.006-.012.014-.017.025-.017 0-.024.006-.041.018-.053l.018-.017c.023-.024.038-.041.044-.053s.014-.018.026-.018c0-.023.003-.038.009-.044l.026-.026c0-.023.006-.038.018-.044.011-.006.017-.015.017-.026l.018-.01c.012-.005.017-.014.017-.025a.408.408 0 00.106-.176.427.427 0 00.053-.062.085.085 0 00.017-.044.425.425 0 00.053-.061.085.085 0 00.017-.044c.141-.235.276-.478.405-.73s.263-.524.404-.817c.094-.2.196-.436.308-.712.11-.275.243-.436.395-.483.27-.106.592-.144.967-.114.375.029.744.044 1.107.044.305 0 .671-.006 1.099-.018.427-.012.694.088.8.299.07.176.029.436-.123.782a7.444 7.444 0 01-.334.694c-.235.352-.466.671-.695.958s-.454.577-.676.87c-.024.024-.039.044-.044.062s-.015.032-.027.044l-.017.008a.028.028 0 00-.018.027c-.023 0-.038.009-.044.026s-.015.032-.026.044c0 .024-.006.038-.018.044s-.017.015-.017.026l-.044.044-.027.027-.061.061-.044.044-.457.624c-.176.24-.258.49-.246.747.023.2.111.372.263.519.153.146.3.29.44.43.34.317.636.6.887.853.252.252.507.554.765.905z"/></svg>',

			'wordpress'   => '<svg width="19" height="19" viewBox="0 0 19 19" xmlns="http://www.w3.org/2000/svg"><path d="M9.9.375c1.242 0 2.408.234 3.498.703 1.09.48 2.042 1.128 2.856 1.943s1.462 1.766 1.943 2.856c.469 1.09.703 2.256.703 3.498s-.234 2.408-.703 3.498c-.48 1.09-1.128 2.042-1.943 2.856s-1.766 1.462-2.856 1.943a8.758 8.758 0 01-3.498.703 8.758 8.758 0 01-3.498-.703 9.307 9.307 0 01-2.856-1.943 9.309 9.309 0 01-1.943-2.856C1.134 11.783.9 10.617.9 9.375s.234-2.408.703-3.498A9.307 9.307 0 013.546 3.02a9.309 9.309 0 012.856-1.943A8.758 8.758 0 019.9.375zm-8.086 9a8.042 8.042 0 001.274 4.35 8.625 8.625 0 001.433 1.697 8.287 8.287 0 001.863 1.23L2.517 6.07c-.234.516-.41 1.052-.527 1.609S1.814 8.8 1.814 9.375zM9.9 17.461c.48 0 .946-.038 1.398-.114.45-.077.875-.19 1.274-.343 0-.024-.006-.044-.018-.062s-.017-.032-.017-.044l-2.496-6.82-2.426 7.067a8.284 8.284 0 002.285.316zm1.125-11.883l2.918 8.684.809-2.672c.175-.563.319-1.055.43-1.477s.167-.809.167-1.16c0-.492-.073-.92-.22-1.283a6.964 6.964 0 00-.448-.932c-.199-.34-.378-.66-.536-.958a1.983 1.983 0 01-.237-.94c0-.375.126-.71.378-1.002.252-.293.571-.44.958-.44h.105a8.386 8.386 0 00-1.16-.879 8.651 8.651 0 00-2.742-1.09A8.44 8.44 0 009.9 1.29c-.703 0-1.383.088-2.04.263-.655.176-1.274.422-1.854.739s-1.11.697-1.59 1.142c-.48.457-.903.961-1.266 1.512h.527c.422 0 .885-.018 1.389-.053l.756-.053c.21-.011.325.088.343.3.017.21-.085.333-.308.368l-.272.027c-.182.017-.396.032-.642.044l2.953 8.754 1.758-5.309-1.266-3.445c-.21-.024-.404-.041-.58-.053l-.263-.018c-.211-.023-.308-.143-.29-.36s.137-.32.36-.308l.764.053c.51.035.958.053 1.345.053.422 0 .89-.018 1.406-.053l.774-.053c.21-.011.322.088.334.3.012.21-.088.333-.299.368l-.273.027c-.181.017-.395.032-.641.044zm2.953 10.793a8.14 8.14 0 001.635-1.266 8.137 8.137 0 001.266-1.634 8.362 8.362 0 00.826-1.934 8.224 8.224 0 00.28-2.162c0-.703-.084-1.383-.254-2.04s-.413-1.265-.73-1.827a4.285 4.285 0 01.07.808c0 .422-.049.87-.149 1.345-.1.475-.26.999-.483 1.573l-2.46 7.137z"/></svg>',

			'yelp'        => '<svg width="16" height="19" viewBox="0 0 16 19" xmlns="http://www.w3.org/2000/svg"><path d="M2.8 2.176a.803.803 0 01.325-.316l.202-.106 3.55-1.3c.07-.011.16-.035.271-.07.112-.035.302.035.571.21.153.083.255.261.308.537l.079.413.035 6.079-.097.369a.63.63 0 01-.465.474.73.73 0 01-.589-.105l-.254-.176-3.655-4.99-.228-.36c-.153-.24-.17-.46-.053-.659zM.938 12.542c-.07-.117-.106-.275-.106-.474v-.3l.246-3.056c0-.082.044-.176.132-.281s.184-.188.29-.246c.117-.082.287-.091.51-.027l.333.097 3.795 1.933.281.193c.188.129.281.31.281.545 0 .316-.058.503-.175.562s-.223.117-.317.176l-4.462.983-.334.08c-.223.052-.38-.01-.474-.185zM7.79 18.34a1.14 1.14 0 01-.422.026l-.21-.026-2.988-.949a.926.926 0 01-.28-.193.7.7 0 01-.176-.334c-.047-.152-.012-.325.105-.518l.176-.29 2.986-3.268.273-.184c.181-.123.36-.132.536-.027.175.082.284.202.325.36l.061.238-.07 4.427-.009.325c-.006.217-.108.354-.307.413zm1.827-9.101l2.39-3.795.175-.237a.537.537 0 01.492-.22c.176 0 .307.053.395.158l.132.158 2.038 2.39.08.228c.052.153.06.31.026.475-.036.152-.156.284-.36.395l-.308.167-4.252 1.23-.342.017c-.229.012-.396-.076-.501-.263-.082-.176-.097-.337-.044-.483l.079-.22zm5.587 4.955l-1.792 2.6-.21.123a.89.89 0 01-.458.123c-.152 0-.307-.082-.465-.246l-.238-.246-2.284-3.83-.087-.317c-.059-.21-.018-.386.123-.527.14-.14.292-.205.456-.193l.246.018 4.287 1.37.3.106c.198.07.298.21.298.421 0 .14-.03.275-.088.404l-.088.194z"/></svg>',

			'youtube'     => '<svg xmlns="http://www.w3.org/2000/svg" width="19" height="14" viewBox="0 0 19 14"><path d="M18.216 3.72l.088.905a20.4 20.4 0 01.088 1.977v1.371c0 .739-.029 1.41-.088 2.013l-.088.905-.158.686c-.105.457-.287.826-.545 1.107-.351.352-.7.566-1.046.642-.345.076-.606.12-.782.132-.62.047-1.3.082-2.039.105-.726.024-1.406.041-2.039.053-.633.012-1.16.018-1.582.018H8.636c-.503-.012-1.092-.024-1.766-.036a116.07 116.07 0 01-2.03-.052 27.164 27.164 0 01-1.565-.088c-.2-.035-.495-.085-.888-.15-.392-.064-.764-.272-1.116-.624-.27-.281-.454-.65-.554-1.107l-.149-.686-.088-.905a20.961 20.961 0 01-.088-2.013v-1.37c0-.727.03-1.39.088-1.987l.088-.897.158-.703c.106-.468.287-.832.545-1.09.352-.375.7-.594 1.046-.659.346-.064.607-.102.782-.114a47.09 47.09 0 012.04-.105c.726-.024 1.406-.041 2.038-.053C7.81.983 8.337.977 8.76.977h1.266c.422 0 .95.006 1.582.018.633.012 1.313.03 2.04.053a47.09 47.09 0 012.038.105c.176.023.437.067.782.132.346.064.695.278 1.046.641.27.27.454.636.554 1.1l.15.693zm-6.398 3.726l.598-.316-4.852-2.532v5.063l4.254-2.215z"/></svg>',

		);

	}
}
